// Purpose         - Makes a polygon planar. It will calculate the average plane of all the 
//                   vertices of the polygon. Then it will align all the points on that 
//                   plane.
// Author          - Marco Biasini ( mbiasini DOT bluewin DOT ch )
// Date            - March 6, 2007
// Feel free to modify

// Modified        - July 11, 2013 by tg_jp

function get_selected_polygon( _object )
{
  var selected = new Array;
    
	for ( var i = 0; i < _object.polygonCount(); ++i )
	{
		if ( _object.polygonSelection( i ) )
		{
        selected.push(i);
		}
	}
	return selected;
}

function get_selected_vertices( _object ) {
  var selected = new Array;
  
  for (var i = 0;i < _object.vertexCount(); i++) {
    if (_object.vertexSelection(i)) {
      selected.push( i );
    }
  }
  return selected;
}

function get_target_polygon( _object ) {
  var selected = new Array;
  
  for( var i = 0;i < _object.polygonCount();i++) {
    
    var polygonSize = _object.polygonSize(i);
    var counter = 0;
    for (var j = 0;j < polygonSize;j++) {
      var vertexIndex = _object.vertexIndex(i,j);
      if (_object.vertexSelection(vertexIndex)) {
        counter++;
      }
    }
    if (counter == 3) {
      selected.push( i );
    }
  }
  
  return selected;
}

function get_dir( _object, pIndex, vIndex, vertices ) {
  var result = false;
  var polygonSize;
  var dir = false;
  
  for (var i = 0;i < _object.polygonCount();i++) {
    if ( i !== pIndex ) {
      polygonSize = _object.polygonSize( i );
      for (var j = 0;j < polygonSize;j++) {
        if (_object.vertexIndex( i, j ) === vIndex) {
          result = i;
          break;  
        }
      }
    }
  }
  
  if (result !== false) {
    polygonSize = _object.polygonSize( result );
    for (var i = 0;i < polygonSize;i++) {
      var tIndex = _object.vertexIndex( result, i );
      if (vertices.indexOf( tIndex ) === -1 && tIndex !== vIndex) {
        dir = _object.vertex( vIndex ).sub( _object.vertex( tIndex ) );
        if (dir.norm() > 1) dir = dir.multiply( 1 / dir.norm() ); 
        break;
      }
    }
  }
  
  return dir;
}

function vec3_dot( _a, _b )
{
	return _a.x*_b.x+_a.y*_b.y+_a.z*_b.z;
}

function vec3_toString( vec ) {
  return "("+vec.x.toFixed(3)+", "+vec.y.toFixed(3)+", "+vec.z.toFixed(3)+")";
}

function main( _doc )
{
	var theObject = _doc.selectedObject();
	
	// Check if there's an object. If not, notify the user.
	if ( !theObject || theObject.family() != NGONFAMILY )
	{
		OS.beep();
		return;
	}
	var polyRep = theObject.core();
		
	if (theObject.parameterWithName) theObject.recordGeometryForUndo(); // not yet.
	
	if (_doc.editMode() === POINT_MODE) {
	  var vertices = get_selected_vertices( polyRep );
	  var poly = get_target_polygon( polyRep );
	  
	  if (vertices.length == 3 && poly.length == 1) {
	    var normal = new Vec3D();
	    var point = new Vec3D();
	    
	    var a = polyRep.vertex( vertices[0] );
	    var b = polyRep.vertex( vertices[1] );
	    var c = polyRep.vertex( vertices[2] );
	    
	    var ab = b.sub( a );
	    var ac = c.sub( a );
	    
	    normal = ab.cross( ac );
	    if (normal.norm() > 1) normal = normal.multiply( 1 / normal.norm() );
	    
	    var polynum = poly[0];
	    var polygonSize = polyRep.polygonSize( polynum );
	    for (var i = 0;i < polygonSize;i++) {
	      var vIndex = polyRep.vertexIndex( polynum, i );
	      
	      if (vertices.indexOf( vIndex ) === -1) {
  	      var v = polyRep.vertex( vIndex );
  	      
  	      var dir = get_dir( polyRep, polynum, vIndex, vertices );
  	      if (dir === false) dir = normal;
  	      
  	      print( vec3_toString( dir ) );
  	      
  	      var pos = triangleIntersect( v, dir, a, b, c );
  	      
  	      if (pos) polyRep.setVertex( vIndex, pos );
    	  }
	    }
	    theObject.update();
	    
	  } else {
	    
	    OS.beep();
	  
	  }
	  
	  
	} else {
  	var poly = get_selected_polygon( polyRep );
  	if ( poly.length > 0 )
  	{
  		var normal = new Vec3D( 0.0, 0.0, 0.0 );
  		var point  = new Vec3D( 0.0, 0.0, 0.0 );
  
  		// Sum them up
  		for (var num in poly) {
              var polynum = poly[num];
              var polySize = polyRep.polygonSize(polynum);
              var normal_sub = new Vec3D( 0.0, 0.0, 0.0 );
              var point_sub  = new Vec3D( 0.0, 0.0, 0.0 );
              for ( var i = 0; i < polySize; ++i )
              {
              	normal_sub = normal_sub.add( polyRep.normal(polynum, i ) );
              	point_sub = point_sub.add( polyRep.vertex( polyRep.vertexIndex( polynum, i ) ) );
              }
              normal_sub = normal_sub.multiply( 1.0/normal_sub.norm() );
              point_sub = point_sub.multiply( 1.0/polySize );
              
              normal = normal.add(normal_sub);
              point = point.add(point_sub);
  		}
  		normal = normal.multiply(1.0/poly.length);
  		point = point.multiply(1.0/poly.length);
  		var planeD = vec3_dot( normal, point );
  		
  		for (var num in poly) {
  		    var polynum = poly[num];
  		    var polySize = polyRep.polygonSize(polynum);
      		for ( var i = 0; i < polySize; ++i )
      		{
      			var v = polyRep.vertex( polyRep.vertexIndex( polynum, i ) );			
      			// Calculate the distance from the plane.
      			var d = -vec3_dot( normal, v )+planeD;
      			// Move the long the planes normal onto the plane.
      			var pos = new Vec3D( v.x+d*normal.x, v.y+d*normal.y, v.z+d*normal.z );
      			polyRep.setVertex( polyRep.vertexIndex( polynum, i ), pos );
      		}
  		}
  		theObject.update();
  		
  	} else {
  	
  		OS.beep();
  	
  	}
  }
}

// Tomas Möller 
var triangleIntersect = function( orig, dir, v0, v1, v2 ) {
	var e1, e2, pvec, tvec, qvec;
	var epsilon = 1e-12;
	var det;
	var t, u, v;
	var inv_det;
	
	e1 = v1.sub( v0 );
	e2 = v2.sub( v0 );
	
	pvec = dir.cross( e2 );
	det = e1.dot( pvec );
	
	if (det.x > epsilon) {
		tvec = orig.sub( v0 );
		u = tvec.dot( pvec );
		//if (u.x < 0 || u.x > det.x) return false;
		
		qvec = tvec.cross( e1 );
		
		v = dir.dot( qvec );
		//if (v.x < 0 || u.x + v.x > det.x) return false; 
	} else if (det.x < -epsilon) {
		tvec = orig.sub( v0 );
		
		u = tvec.dot( pvec );
		//if (u.x > 0 || u.x < det.x) return false;
		
		qvec = tvec.cross( e1 );
		
		v = dir.dot( qvec );
		//if (v.x > 0 || u.x + v.x < det.x) return false;
	} else {
		return false;
	}
	
	inv_det = 1 / det.x;
	
	t = e2.dot( qvec );
	t = t.x * inv_det;
	
	u = u.x * inv_det;
	v = v.x * inv_det;
	
	var crossPoint = orig.add( dir.multiply( t ) );
	
	return crossPoint;
}

