/*
  LuxCheetah.js
  
  Exporter Script for Cheetah3D.
  
  http://www.luxrender.net/wiki/index.php/Scene_file_format
*/

// global variables.
var target_version_num = "0.6RC4";
var version_num = "0.1.1";
var cache_name = "chCache_";

//
Object.toString = function() {
	return '[name:'+ this.getParameter("name") + ', type:' + this.type() + ']';
};
Vec2D.prototype.toString = function() {
	return this.u.toFixed(6) + ' ' + this.v.toFixed(6);
}
Vec2D.prototype.toUV = function() {
	return this.x.toFixed(6) + ' ' + this.y.toFixed(6);
}
Vec3D.prototype.toString = function() {
	return this.x.toFixed(6) + ' ' + this.y.toFixed(6) + ' ' + this.z.toFixed(6);
}
Vec3D.prototype.normalize = function() {
	var l = this.norm();
	return (l < 0.000001 /* == 0 */ ) ? new Vec3D() : this.multiply(1/l);
}
Vec4D.prototype.toString = function() {
	return this.x.toFixed(6) + ' ' + this.y.toFixed(6) + ' ' + this.z.toFixed(6) + ' ' + this.w.toFixed(6);
}
Vec4D.prototype.toRGB = function() {
	return this.x.toFixed(6) + ' ' + this.y.toFixed(6) + ' ' + this.z.toFixed(6);
}
Vec4D.prototype.toUV = function() {
	return this.x.toFixed(6) + ' ' + this.y.toFixed(6);
}
Mat4D.prototype.toString = function() {
	return this.m00.toFixed(6) + ' ' + this.m10.toFixed(6) + ' '+ this.m20.toFixed(6) + ' ' + this.m30.toFixed(6) + '  ' +
				this.m01.toFixed(6) + ' ' + this.m11.toFixed(6) + ' '+ this.m21.toFixed(6) + ' ' + this.m31.toFixed(6) + '  ' +
				this.m02.toFixed(6) + ' ' + this.m12.toFixed(6) + ' '+ this.m22.toFixed(6) + ' ' + this.m32.toFixed(6) + '  ' +
				this.m03.toFixed(6) + ' ' + this.m13.toFixed(6) + ' '+ this.m23.toFixed(6) + ' ' + this.m33.toFixed(6);
}
//
var mesh_list = [];
var vol_list = [];
var portal_list = [];
var material_list = [];
var light_list = [];

var intSize = 0;

var final_mat = new Mat4D(ROTATE_HPB, 0, 90, 0);
var environment = undefined;

// global var
var gGamma;

// console path
// 
var luxconsole_file = ".luxcheetah_consolepath.txt";
var luxapp_file = ".luxcheetah_apppath.txt";
var luxconsole = "";
var luxapp = "";

function buildUI( tool ) {
	
	//tool.addParameterButton("test sundir", "test", "testSundir");
	
	tool.addParameterSeparator("Export");
	tool.addParameterButton("export .lxs", "export", "exportLXS");
	
	tool.addParameterSeparator("Render");
	tool.addParameterBool("do render", 0, 0, 1, false, false);
	tool.addParameterBool("use luxconsole", 0, 0, 1, false, false);
	
	tool.addParameterSeparator("Animation");
	tool.addParameterBool("export animation", 0, 0, 1, false, false);
	tool.addParameterInt("fps", 30, 1, 1000, false, false);
	tool.addParameterFloat("start time", 0, 0, 100000, false, false);
	tool.addParameterFloat("end time", 4, 0, 100000, false, false);
	
	tool.addParameterSeparator("General Settings");
	tool.addParameterBool("split light .lxl", 0, 0, 1, false, false);
	tool.addParameterBool("split material .lxm", 0, 0, 1, false, false);
	tool.addParameterBool("split geom .lxo", 0, 0, 1, false, false);
	tool.addParameterBool("split volume .lxv", 0, 0, 1, false, false);
	
	tool.addParameterSeparator("Global Settings");
	tool.addParameterFloat("imagemap gamma", 2.2, 1.0, 6.0, false, false);
	
	// calc intSize
	if (intSize == 0) {
		var cache = new File("/tmp/"+cache_name+"_hol.dat");
		cache.open(WRITE_MODE);
		cache.writeInt(-1);
		cache.close();
		
		intSize = cache.size(); // getting 'int' byte size ( must be 4 )
		OS.system('rm ' + cache.directory() + '/' + cache.lastPathComponent());
	}
}

function exportLXS( tool ) {
	var doc = tool.document();
	
	// general parameters
	var sep_light = tool.getParameter("split light .lxl");
	var sep_mat = tool.getParameter("split material .lxm");
	var sep_geom = tool.getParameter("split geom .lxo");
	var sep_vol = tool.getParameter("split volume .lxv");
	
	gGamma = tool.getParameter("imagemap gamma");
	
	var export_anim = tool.getParameter("export animation");
	if (export_anim) {
		var fps = tool.getParameter("fps");
		var time_s = tool.getParameter("start time");
		var time_e = tool.getParameter("end time");
		if (time_e > doc.animationEnd()) time_e = doc.animationEnd();
		
		var dt = 1/fps;
		var frame = -1;
	} else {
		var time_s = -1;
		var time_e = 1;
	}
	
	if (tool.getParameter("do render")) {
		OS.system("echo $HOME > /tmp/luxcheetah_homecheck");
		var home_check = new File("/tmp/luxcheetah_homecheck");
		home_check.open(READ_MODE);
		if (home_check.isOpen()) {
			var home_dir = home_check.readln();
			home_check.close();
		} else {
			return;
		}
		var valid_path = false;
		if (tool.getParameter('use luxconsole') || tool.getParameter("export animation")) {
			var console_path = new File(home_dir + '/' + luxconsole_file);
			if (console_path.exist()) {
				console_path.open(READ_MODE);
				luxconsole = console_path.readln();
				console_path.close();
				var check_console = new File(luxconsole);
				if (check_console.exist()) valid_path = true;
			}
			if (valid_path == false) {
				var msg_check = OS.messageBox("Select 'luxconsole'","To 'do render' directly from Cheetah3D, please select 'luxconsole'.");
				var luxconsole = OS.runOpenPanel("");
				if (luxconsole == null) return;
				console_path.open(WRITE_MODE);
				console_path.write(luxconsole);
				console_path.close();
			}
		} else {
			var app_path = new File(home_dir + '/' + luxapp_file);
			if (app_path.exist()) {
				app_path.open(READ_MODE);
				luxapp = app_path.readln();
				app_path.close();
				var check_app = new File(luxapp);
				if (check_app.exist()) valid_path = true;
			}
			if (valid_path == false) {
				var msg_check = OS.messageBox("Select 'luxrender.app'","To 'do render' directly from Cheetah3D, please select 'luxrender.app'.");
				var luxapp = OS.runOpenPanel("app");
				if (luxapp == null) return;
				app_path.open(WRITE_MODE);
				app_path.write(luxapp);
				app_path.close();
			}
		}
	}
	
	print('--- LuxCheetah exports .lxs ---');
	var path = OS.runSavePanel("lxs");
	if ( path == null ) return;
	
	var gLxs_file = new File(path);
	var gLxs_filename = gLxs_file.lastPathComponent();
	var gBase_filename = gLxs_filename.substr(0, gLxs_filename.length-4);
	var cam = doc.activeCamera();
	
	var lxs_list = [];
	
	while ( time_s <= time_e ) {
		frame++;
		
		if (export_anim) {
			var frame_str = "" + frame;
			while(frame_str.length < 6) frame_str = '0'+frame_str;
			
			var lxs_file = new File(gLxs_file.directory() + '/' + gBase_filename + '_' + frame_str + '.lxs');
			var lxs_filename = lxs_file.lastPathComponent();
			var base_filename = lxs_filename.substr(0, lxs_filename.length-4);
			
			doc.setAnimPosition(time_s);
			time_s += dt;
		} else {
			var lxs_file = gLxs_file;
			var base_filename = gBase_filename;
			
			time_s = time_e + 1; // to stop loop
		}
		
		lxs_list.push(lxs_file.directory() + '/' + lxs_file.lastPathComponent());
		print("exporting for "+lxs_list[ lxs_list.length - 1 ]);
		// initialize
		mesh_list.length = 0; material_list.length = 0; vol_list.length = 0; portal_list.length = 0; light_list.length = 0;
		environment = undefined;
		
		storeObjects( doc.root(), tool );
		
		lxs_file.open(WRITE_MODE);
		
		// start writing.
		lxs_file.writeln("# Lux Render v" + target_version_num + " Scene File");
		if (export_anim) lxs_file.writeln("# for frame "+frame_str);
		lxs_file.writeln("# Exported by LuxCheetah v." + version_num + "\n");
		
		
		lxs_file.writeln(cameraLine(tool, cam));
		
		lxs_file.write(filmLine(tool, cam, lxs_file.directory() + '/' + base_filename));
		
		lxs_file.write(pixelFilterLine(tool, cam));
		lxs_file.write(samplerLine(tool, cam));
		lxs_file.write(surfaceIntegratorLine(tool, cam));
		lxs_file.write(volumeIntegratorLine(tool, cam));
		lxs_file.write(acceleratorLine(tool, cam));
		
		// world start
		
		lxs_file.writeln("\nWorldBegin\n");
		
		var i,j,k,iter;
		var len;
		
		//
		// light block
		//
		lxs_file.writeln("# light");
		if (sep_light) {
			var lxl_file = new File(lxs_file.directory() + '/' + base_filename + '-light.lxl');
			lxl_file.open(WRITE_MODE);
			lxs_file.writeln("Include \"" + base_filename + "-light.lxl\"\n");
		} else {
			var lxl_file = lxs_file;
		}
		var portalshape = '';
		// check portal
		if (portal_list.length > 0) {
			portalshape += "\tPortalShape \"trianglemesh\" \"integer indices\" [\n";
			var portalpoint = "";
			var tri_index = 0;
			
			for (i = 0;i < portal_list.length;i++) {
				var obj = portal_list[i];
				var mat = obj.obj2WorldMatrix();
				var core = obj.modCore();
				var polyCount = core.polygonCount();
				var indices = [];
				
				for (j = 0;j < polyCount;j++) {
					var polySize = core.polygonSize( j );
					for (k = 0;k < polySize - 2;k++) {
						indices = core.triangle( j, k );
						
						portalshape += "\t\t"+(tri_index*3)+" "+(tri_index*3+1)+" "+(tri_index*3+2)+"\n";
						portalpoint += "\t\t"+mat.multiply( core.vertex( core.vertexIndex( j, indices[0]) ) )+"\n";
						portalpoint += "\t\t"+mat.multiply( core.vertex( core.vertexIndex( j, indices[1]) ) )+"\n";
						portalpoint += "\t\t"+mat.multiply( core.vertex( core.vertexIndex( j, indices[2]) ) )+"\n";
						
						tri_index++;
					}
				}
			}
			portalshape += "\t] \"point P\" [\n";
			portalshape += portalpoint + "\n\t]";
		}
		// check camera light, shoul be use flash light after integrating.
		if (cam.getParameter("cameraLightOnOff")) {
			lxl_file.writeln("AttributeBegin # camera light");
			lxl_file.writeln("\tTransform ["+final_mat.multiply(cam.obj2WorldMatrix()).toString()+"]");
			lxl_file.writeln("\tLightGroup \"Light::Camera\"");
			lxl_file.writeln("\tLightSource \"point\""+paramLine('L', new Vec4D(1,1,1,1), 'color')+paramLine('from', new Vec3D(0,0,0), 'point'));
			lxl_file.writeln("AttributeEnd\n");
		}
		// check IBL panorama EXR
		var hdriTag = tagForType( cam, HDRITAG);
		if (hdriTag) { // 
			var hdriBg = hdriTag.getParameter("hdriBackgroundImage");
			//var hdriRot = hdriTag.getParameter("hdriRotate");
			var mappingTypes = ['angular','vcross','latlong'];
			var hdriMapping = mappingTypes[ parseInt(hdriTag.getParameter("hdriMapping")) ];
			var hdriRot = hdriTag.getParameter("hdriRotate");
			
			var hdriLine = "\tLightSource \"infinite\"";
			
			if (hdriBg != 'none') {
				hdriLine += paramLine("mapping", hdriMapping, "string") +
								paramLine('mapname', hdriBg, 'string') +
								paramLine('gamma', 1.0, 'float');
			} else {
				hdriLine += paramLine('L', cam.getParameter("backgroundColor"), 'color');
			}
			hdriLine += paramLine('gain', hdriTag.getParameter("hdriPower"), 'float');
			
			lxl_file.writeln("AttributeBegin # HDRI light");
			if (hdriRot != 0) lxl_file.writeln("\tRotate "+hdriRot+" 0 0 1\n");
			lxl_file.writeln("\tLightGroup \"Light::HDRI\"");
			lxl_file.writeln( hdriLine );
			if (hdriRot != 0 && portalshape != '') lxl_file.writeln("\tRotate -"+hdriRot+" 0 0 1\n");
			lxl_file.writeln( portalshape );
			lxl_file.writeln("AttributeEnd\n");
		}
		len = light_list.length;
		for (i =  0;i < len;i++) {
			lxl_file.writeln( lightLine(tool, light_list[i], i, portalshape) );
		}
		if (sep_light) lxl_file.close();
		//
		// volume block
		//
		lxs_file.writeln("# volume");
		if (sep_vol) {
			var lxv_file = new File(lxs_file.directory() + '/' + base_filename + '-vol.lxv');
			lxv_file.open(WRITE_MODE);
			lxs_file.writeln("Include \"" + base_filename + "-vol.lxv\"\n");
		} else {
			var lxv_file = lxs_file;
		}
		len = vol_list.length;
		for (i = 0;i < len;i++) {
			writeVolumeLine(tool, vol_list[i], i, lxv_file);
		}
		if (sep_vol) lxv_file.close();
		//
		// material block
		//
		lxs_file.writeln("# material");
		if (sep_mat) {
			var lxm_file = new File(lxs_file.directory() + '/' + base_filename + '-mat.lxm');
			lxm_file.open(WRITE_MODE);
			lxs_file.writeln("Include \"" + base_filename + "-mat.lxm\"\n");
		} else {
			var lxm_file = lxs_file;
		}
		/// defualt material
		lxm_file.writeln('MakeNamedMaterial "luxCh_default" "string type" ["matte"]'+paramLine('Kd', new Vec4D(0.75, 0.75, 0.75, 1.0), 'color')+"\n");
		iter = 0;
		len = material_list.length;
		for (i = 0;i < len;i++) {
			var mat_len = material_list[i].length;
			for (j = 0;j < mat_len;j++) {
				writeMaterialLine(tool, material_list[i][j], iter, lxm_file);
				iter++;
			}
		}
		if (sep_mat) lxm_file.close();
		//
		// geometry block
		//
		lxs_file.writeln("# geometry");
		if (sep_geom) {
			var lxo_file = new File(lxs_file.directory() + '/' + base_filename + '-geom.lxo');
			lxo_file.open(WRITE_MODE);
			lxs_file.writeln("Include \"" + base_filename + "-geom.lxo\"\n");
		} else {
			var lxo_file = lxs_file;
		}
		len = mesh_list.length;
		for (i = 0;i < len;i++) {
			writeMeshLine(tool, mesh_list[i], i, lxo_file, lxs_file);
		}
		if (sep_geom) lxo_file.close();
		//
		lxs_file.writeln("WorldEnd\n");
		
		lxs_file.close();
	}
	
	if (tool.getParameter("do render")) {
		if (tool.getParameter("use luxconsole") || tool.getParameter("export animation")) {
			var console = new File(luxconsole);
			if (luxconsole != '' && console.exist()) {
				if ( lxs_list.length == 1 ) { // if it isn't animation render, execute console as background
					print("rendering for "+lxs_list[0]);
					OS.system(luxconsole+" "+lxs_list[0]+" &");
				} else {
					for (i = 0;i < lxs_list.length;i++) {
						print("rendering for "+lxs_list[i]);
						OS.system(luxconsole+" "+lxs_list[i]);
					}
				}
			} else {
				OS.messageBox("luxconsole is not found!!","Please make sure 'luxconsole'.");
			}
		} else {
			var app = new File(luxapp + '/Contents/MacOS/luxrender');
			if (luxapp != '' && app.exist()) {
				print("rendering for "+lxs_list[0]);
				OS.system(luxapp+"/Contents/MacOS/luxrender "+lxs_list[0]+" &");
			} else {
				OS.messageBox("luxrender.app is not found!!", "Please make sure 'luxrender.app'.");
			}
		}
	}
}

function paramLine(val_name, val, val_type) {
	if (val_type == "float[]") {
		var str = "\n   " + '"float '+val_name+'" ';
	} else {
		var str = "\n   " + '"'+val_type+' '+val_name+'" ';
	}
	switch (val_type) {
		case 'integer':
			str += '['+parseInt(val)+']';
			break;
		case 'float':
			str += '['+val.toFixed(6)+']';
			break;
		case 'float[]':
			str += '[';
			var len = val.length;
			for (var i = 0;i < len;i++) {
				str += val[i].toFixed(6);
				if (i < len -1) str += ' '; // not last val
			}
			str += ']';
			break;
		case 'point':
			str += '['+val+']'; // val must be Vec3D
			break;
		case 'vector':
			str += '['+val.normalize()+']'; // val must be Vec3D
			break;
		case 'normal':
			str += '['+val+']'; // val must be Vec3D?
			break;
		case 'color':
			str += '['+val.toRGB()+']'; // val must be Vec4D
			break;
		case 'bool':
			if (val) {
				str += '["true"]';
			} else {
				str += '["false"]';
			}
			break;
		case 'string':
			str += '["'+val.replace('"', '')+'"]';
			break;
		case 'texture':
			str += '["'+val+'"]';
	}
	return str;
}

// LuxTag Camera.js
var camera_distributions = ["uniform", "exponential", "inverse exponential", "gaussian", "inverse gaussian"];

function cameraLine(tool, cam) {
	
	var mat = cam.obj2WorldMatrix();
	var pos = final_mat.multiply( cam.getParameter("position") );
	var foc = final_mat.multiply( mat.multiply(new Vec3D( 0,  0, -1)) );
	
	var up = mat.multiply(new Vec3D(0, 1, 0)); // 
	up = up.sub(new Vec3D( mat.m03, mat.m13, mat.m23)); // translate back.
	up  = final_mat.multiply( up ).normalize(); // up-vector requires normalization.
	
	var tagAlt = tagForScriptName(cam, "LuxTag Camera.js");
	var tagDof = tagForType(cam, DOFTAG);
	
	var checkShift = false;
	var scale = 1.0;
	
	var str = "LookAt "+pos.toString()+" "+foc.toString()+" "+up.toString()+"\n";
	switch (parseInt(cam.getParameter("projection"))) {
		case 0:
			var cam_type = 'perspective';
			str += 'Camera "'+cam_type+'"';
			var fov = cam.getParameter("fieldOfView");
			var cam_resX = cam.getParameter("resolutionX");
			var cam_resY = cam.getParameter("resolutionY");
			if (cam_resY > cam_resX) { // horizontal -> vertical
				fov = fov / (180/Math.PI);
				fov = 2.0 * ( Math.tan( fov/2 ) * (cam_resX / cam_resY) );
				fov *= (180 / Math.PI);
			}
			str += paramLine("fov", fov, "float");
			checkShift = true;
			
			break;
		case 1:
			var cam_type = 'environment';
			str += 'Camera "'+cam_type+'"';
			break;
		case 2:
			var cam_type = 'orthographic';
			str += 'Camera "'+cam_type+'"';
			checkShift = true;
			
			if (tagAlt) {
				str += paramLine("scale", tagAlt.getParameter("ortho scale"), "float");
				scale = tagAlt.getParameter("scale") / 2;
			}
			break;
	}
	// DOF
	if (( cam_type == 'environment' || cam_type == 'perspective' ) && tagDof) {
		str += paramLine("lensradius", tagDof.getParameter("lensRadius"), 'float');
		if (tagAlt) {
			if (tagAlt.getParameter("autofocus")) {
				str += paramLine("autofocus", tagAlt.getParameter("autofocus"), 'bool');
			} else {
				str += paramLine("focaldistance", tagDof.getParameter("focalDistance"), 'float');
			}
			if (cam_type == 'perspective') {
				str += paramLine("blades", tagAlt.getParameter("blades"), "integer");
				str += paramLine("distribution", camera_distributions[ parseInt(tagAlt.getParameter("distribution")) ], "string");
				str += paramLine("power", tagAlt.getParameter("power"), "integer");
			}
		} else {
			str += paramLine("focaldistance", tagDof.getParameter("focalDistance"), 'float');
		}
	}
	// Shift
	if (checkShift && tagAlt && tagAlt.getParameter("use shift")) {
		var shiftX = tagAlt.getParameter("shift X");
		var shiftY = tagAlt.getParameter("shift Y");
		if (tagAlt.getParameter("use aspect ratio")) {
			var ratio = 1 / tagAlt.getParameter("aspect ratio");
			str += paramLine("framescpectratio", tagAlt.getParameter("aspect ratio"), "float");
		} else {
			var ratio = cam.getParameter("resolutionY") / cam.getParameter("resolutionX");
		}
		if (ratio < 1.0) {
			var screenwindow = [ ( 2 * shiftX - 1) * scale, ( 2 * shiftX + 1) * scale, ( 2 * shiftY - ratio) * scale, ( 2 * shiftY + ratio) * scale ];
		} else {
			var screenwindow = [ ( 2 * shiftX - 1/ratio) * scale, ( 2 * shiftX + 1/ratio) * scale, ( 2 * shiftY - 1) * scale, (2 * shiftY + 1) * scale ];
		}
		str += paramLine("screenwindow", screenwindow, "float[]");
	}
	// Motion Blur
	if (tagAlt && tagAlt.getParameter("use motion blur")) {
		str += paramLine("shutteropen", 0.0, "float");
		str += paramLine("shutterclose", 1.0, "float");
	}
	// Clip
	if (tagAlt == false || tagAlt.getParameter("use near/far clip")) {
		str += paramLine("hither", cam.getParameter("clipNear"), 'float');
		str += paramLine("yon", cam.getParameter("clipFar"), 'float');
	}
	
	return str + "\n";
}

// 
var tonemap_kernels = ["reinhard","linear","contrast","maxwhite"];
var exr_channels = ['RGB', 'RGBA', 'YA', 'Y'];
var exr_compressiontypes = ['PIZ (lossless)', 'RLE (lossless)', 'ZIP (lossless)', 'Pxr24 (lossy)', 'None'];
var exr_normalizationtypes = ["None", "Min/Max", "Camera Start/End clip"];
var png_channels = ['RGB', 'RGBA', 'YA', 'Y'];
var tga_channels = ['RGB', 'RGBA', 'Y'];

function filmLine(tool, cam, filename) {
	var tag = tagForScriptName(cam, "LuxTag Film.js");
	
	str = '';
	
	str += 'Film "fleximage"';
	str +=  paramLine("xresolution", cam.getParameter("resolutionX"), "integer") + 
			paramLine("yresolution", cam.getParameter("resolutionY"), "integer") + 
			paramLine("filename", filename, 'string');
	if (tag) {
		var tonemapkernel = tonemap_kernels[ parseInt( tag.getParameter("tonemap kernel") ) ];
		
		str +=  paramLine("haltspp", tag.getParameter("haltspp"), 'integer') +
				paramLine("premultiplyalpha", tag.getParameter("premultiplyalpha"), 'bool') +
				paramLine("tonemapkernel", tonemapkernel, 'string');
		switch( tonemapkernel ) {
			case 'reinhard':
				str +=  paramLine("reinhard_autoywa", tag.getParameter("reinhard_autoywa"), 'bool');
				if (! tag.getParameter("reinhard_autoywa")) {
					str += paramLine("reinhard_ywa", tag.getParameter("reinhard_ywa"), 'float');
				}
				str += paramLine("reinhard_prescale", tag.getParameter("reinhard_prescale"), 'float') + 
						 paramLine("reinhard_postscale", tag.getParameter("reinhard_postscale"), 'float') + 
						 paramLine("reinhard_burn", tag.getParameter("reinhard_burn"), 'float');
				break;
			case 'linear':
				str +=  paramLine("linear_sensitivity", tag.getParameter("linear_sensitivity"), 'float') + 
						paramLine("linear_exposure", tag.getParameter("linear_exposure"), 'float') + 
						paramLine("linear_fstop", tag.getParameter("linear_fstop"), 'float') + 
						paramLine("linear_gamma", tag.getParameter("linear_gamma"), 'float');
				break;
			case 'contrast':
				str +=  paramLine("contrast_ywa", tag.getParameter("contrast_ywa"), 'float');
				break;
			case 'maxwhite':
				break;
		}
		str +=  paramLine("displayinterval", tag.getParameter("displayinterval"), 'integer') +
				paramLine("writeinterval", tag.getParameter("writeinterval"), 'integer');
		// exr
		str +=  paramLine("write_exr", tag.getParameter("write_exr"), 'bool') +
				paramLine("write_exr_channels", exr_channels[ parseInt(tag.getParameter("write_exr_channels")) ], "string") +
				paramLine("write_exr_halftype", tag.getParameter("write_exr_halftype"), "bool") +
				paramLine("write_exr_compressiontype", exr_compressiontypes[ parseInt(tag.getParameter("write_exr_compressiontype")) ], "string") +
				paramLine("write_exr_applyimaging", tag.getParameter("write_exr_applyimaging"), "bool") +
				paramLine("write_exr_gamutclamp", tag.getParameter("write_exr_gamutclamp"), "bool") +
				paramLine("write_exr_ZBuf", tag.getParameter("write_exr_ZBuf"), "bool") +
				paramLine("write_exr_zbuf_normalizationtype", exr_normalizationtypes[ parseInt(tag.getParameter("write_exr_zbugf_normalizationtype")) ], "string");
		// png
		str +=  paramLine("write_png", tag.getParameter("write_png"), "bool") +
				paramLine("write_png_channels", png_channels[ parseInt(tag.getParameter("write_png_channels")) ], "string") +
				paramLine("write_png_16bit", tag.getParameter("write_png_16bit"), "bool") +
				paramLine("write_png_gamutclamp", tag.getParameter("write_png_gamutclamp"), "bool");
		// tga
		str +=  paramLine("write_tga", tag.getParameter("write_tga"), "bool") +
				paramLine("write_tga_channels", tga_channels[ parseInt(tag.getParameter("write_tga_channels")) ], "string") +
				paramLine("write_tga_gamutclamp", tag.getParameter("write_tga_gamutclamp"), "bool");
		// resume & debug
		str +=  paramLine("write_resume_flm", tag.getParameter("write_resume_flm"), 'bool') +
				paramLine("restart_resume_flm", tag.getParameter("restart_resume_flm"), 'bool') +
				paramLine("reject_warmup", tag.getParameter("reject_warmup"), 'integer') +
				paramLine("debug", tag.getParameter("debug"), 'bool');
		// colorspace
		str +=  paramLine("colorspace_white", [tag.getParameter("cspacewhiteX"), tag.getParameter("cspacewhiteY")], 'float[]') +
				paramLine("colorspace_red", [tag.getParameter("cspaceredX"), tag.getParameter("cspaceredY")], 'float[]') +
				paramLine("colorspace_green", [tag.getParameter("cspacegreenX"), tag.getParameter("cspacegreenY")], 'float[]') +
				paramLine("colorspace_blue", [tag.getParameter("cspaceblueX"), tag.getParameter("cspaceblueY")], 'float[]') +
				paramLine("gamma", tag.getParameter("gamma"), "float");
	}
	
	return str + "\n\n";
}

// LuxTag PixelFilter.js
var pixelFilter_types = ['mitchell', 'box', 'gaussian', 'sinc', 'triangle'];

function pixelFilterLine(tool, cam) {
	var tag = tagForScriptName(cam, "LuxTag PixelFilter.js");
	
	if (tag) {
		var filter = pixelFilter_types[ parseInt(tag.getParameter("pixel filter type")) ];
		var str = 'PixelFilter "'+filter+'"';
		switch (filter) {
			case 'mitchell':
				str += paramLine("xwidth", tag.getParameter("xwidth"), 'float') +
						paramLine("ywidth", tag.getParameter("ywidth"), 'float');
				var c = tag.getParameter("mitchell::sharpness") * 0.5;
				var b = 1 - tag.getParameter("mitchell::sharpness");
				str += paramLine('B', b, 'float') + 
						paramLine('C', c, 'float');
				break;
			case 'box':
				str += paramLine("xwidth", tag.getParameter("box::xwidth"), 'float') +
						paramLine("ywidth", tag.getParameter("box::ywidth"), 'float');
				break;
			case 'gaussian':
				str += paramLine("xwidth", tag.getParameter("xwidth"), 'float') +
						paramLine("ywidth", tag.getParameter("ywidth"), 'float');
				str += paramLine('alpha', tag.getParameter("gaus::alpha"), 'float');
				break;
			case 'sinc':
				str += paramLine('tau', tag.getParameter("sinc::tau"), 'float') + 
						paramLine('xwidth', tag.getParameter("sinc::xwidth"), 'float') +
						paramLine('ywidth', tag.getParameter("sinc::ywidth"), 'float');
				break;
			case 'triangle':
				str += paramLine("xwidth", tag.getParameter("xwidth"), 'float') +
						paramLine("ywidth", tag.getParameter("ywidth"), 'float');
				break;
		}
		return str + "\n";
	} else {
		return '';
	}
}

// LuxTag Sampler.js
var samplers = ["metropolis","erpt","lowdiscrepancy","random"];
var pixel_samplers = ["linear", "tile", "random", "vegas", "lowdiscrepancy", "hilbert"];

function samplerLine(tool, cam) {
	var tag = tagForScriptName(cam, "LuxTag Sampler.js");
	
	if (tag) {
		var sampler = samplers[parseInt(tag.getParameter("sampler"))];
		var str = 'Sampler "' + sampler + '"';
		
		switch(sampler) {
			case "random":
				str += paramLine('pixelsampler', pixel_samplers[parseInt(tag.getParameter("ram::pixelsampler"))], 'string');
				str += paramLine('xsamples', tag.getParameter("ram::xsamples"), 'integer');
				str += paramLine('ysamples', tag.getParameter("ram::ysamples"), 'integer');
				break;
			case "lowdiscrepancy":
				str += paramLine('pixelsampler', pixel_samplers[parseInt(tag.getParameter("low::pixelsampler"))], 'string');
				str += paramLine('pixelsamples', tag.getParameter("low::pixelsamples"), 'integer');
				break;
			case "metropolis":
				if (tag.getParameter("metr::use advanced")) {
					str += paramLine('largemutationprob', tag.getParameter("metr::largemutationprob"), 'float');
					str += paramLine('maxconsecrejects', tag.getParameter("metr::maxconsecrejects"), 'integer');
					str += paramLine('initsamples', tag.getParameter("metr::initsamples"), 'integer');
					str += paramLine('stratawidth', tag.getParameter("metr::stratawidth"), 'integer');
					str += paramLine('usevariance', tag.getParameter("metr::usevariance"), 'bool');
				} else {
					str += paramLine("largemutationprob", 1 - tag.getParameter("metr::strength"), 'float');
				}
				break;
			case "erpt":
				str += paramLine('initsamples', tag.getParameter("erpt::initsamples"), 'integer');
				str += paramLine('chainlength', tag.getParameter("erpt::chainlength"), 'integer');
				str += paramLine('mutationrange', tag.getParameter("erpt::stratawidth"), 'integer');
				break;
		}
		
		return str+"\n\n";
	} else {
		return '';
	}
}

// LuxTag Surface Integrator.js
var surface_integrator_types = ["directlighting", "path", "bidirectional", "exphotonmap", "distributedpath" ];
var si_strategy = ["auto", "all", "one"];
var rr_strategy = ["efficiency", "probability", "none"];

var ep_renderingmode = ["directlighting", "path"];

function surfaceIntegratorLine( tool, cam) {
	var tag = tagForScriptName(cam, "LuxTag Surface Integrator.js");
	
	if (tag) { // 
		var integ = surface_integrator_types[ parseInt(tag.getParameter("surface integrator type")) ];
		var str = 'SurfaceIntegrator "'+integ+'"';
		
		switch(integ) {
			case "directlighting":
				str += paramLine("maxdepth", tag.getParameter("dl::maxdepth"), "integer") + 
						paramLine("strategy", si_strategy[ parseInt(tag.getParameter("dl::strategy")) ], "string");
				break;
				
			case "path":
				str +=  paramLine("maxdepth", tag.getParameter("path::maxdepth"), "integer") + 
						paramLine("strategy", si_strategy[ parseInt(tag.getParameter("path::strategy")) ], "string") +
						paramLine("rrstrategy", rr_strategy[ parseInt(tag.getParameter("path::rrstrategy")) ], "string");
				if (rr_strategy[ parseInt( tag.getParameter("path::rrstrategy") ) ] == "probability") {
					str += paramLine("rrcontinueprob", tag.getParameter("path::rrcontinueprob"), "float");
				}
				str += paramLine("includeenvironment", tag.getParameter("path::includeenviron"), "bool");
				break;
				
			case "bidirectional":
				str += paramLine("eyedepth", tag.getParameter("bd::eyedepth"), "integer") + paramLine("lightdepth", tag.getParameter("bd::lightdepth"), "integer");
				str += paramLine("strategy", si_strategy[ parseInt(tag.getParameter("bd::strategy")) ], "string");
				break;
				
			case "exphotonmap":
				str += paramLine("indirectphotons", tag.getParameter("ep::indirectphotons"), "integer") +
						 paramLine("maxdirectphotons", tag.getParameter("ep::maxdirectphotons"), "integer") +
						 paramLine("causticphotons", tag.getParameter("ep::causticphotons"), "integer") + 
						 paramLine("strategy", si_strategy[ parseInt(tag.getParameter("ep::strategy")) ], "string") +
						 paramLine("maxdepth", tag.getParameter("ep::maxdepth"), "integer") +
						 paramLine("maxdist", tag.getParameter("ep::maxdist"), "float") +
						 paramLine("nused", tag.getParameter("ep::nused"), "integer") +
						 paramLine("renderingmode", ep_renderingmode[ parseInt(tag.getParameter("ep::renderingmode")) ], "string") + 
						 paramLine("finalgather", tag.getParameter("ep::finalgather"), "bool");
						
						if (tag.getParameter("ep::finalgather")) {
							str += paramLine("finalgathersamples", tag.getParameter("ep::finalgathersamples"), "integer") +
									 paramLine("gatherangle", tag.getParameter("ep::gatherangle"), "float") +
									 paramLine("gatherrrstrategy", rr_strategy[ parseInt(tag.getParameter("ep::gatherrrstrategy")) ], "string") +
									 paramLine("gatherrrcontinueprob", tag.getParameter("ep::gatherrrcontinueprob"), "float");
						}
						
				break;
			case "distributedpath":
				str += paramLine("strategy", si_strategy[ parseInt(tag.getParameter("dp::strategy")) ], "string");
				str += paramLine("directsampleall", tag.getParameter("dp::directsampleall"), "bool");
				str += paramLine("directsamples", tag.getParameter("dp::directsamples"), "integer");
				str += paramLine("indirectsampleall", tag.getParameter("dp::indirectsampleall"), "bool");
				str += paramLine("indirectsamples", tag.getParameter("dp::indirectsamples"), "integer");
				
				str += paramLine("diffusereflectdepth", tag.getParameter("dp::diffusereflectdepth"), "integer");
				str += paramLine("diffusereflectsamples", tag.getParameter("dp::diffusereflectsamples"), "integer");
				str += paramLine("diffuserefractdepth", tag.getParameter("dp::diffuserefractdepth"), "integer");
				str += paramLine("diffuserefractsamples", tag.getParameter("dp::diffuserefractsamples"), "integer");
				str += paramLine("directdiffuse", tag.getParameter("dp::directdiffuse"), "bool");
				str += paramLine("indirectdiffuse", tag.getParameter("dp::indirectdiffuse"), "bool");
				
				str += paramLine("glossyreflectdepth", tag.getParameter("dp::glossyreflectdepth"), "integer");
				str += paramLine("glossyreflectsamples", tag.getParameter("dp::glossyreflectsamples"), "integer");
				str += paramLine("glossyrefractdepth", tag.getParameter("dp::glossyrefractdepth"), "integer");
				str += paramLine("glossyrefractsamples", tag.getParameter("dp::glossyrefractsamples"), "integer");
				str += paramLine("directglossy", tag.getParameter("dp::directglossy"), "bool");
				str += paramLine("indirectglossy", tag.getParameter("dp::indirectglossy"), "bool");
				
				str += paramLine("specularreflectdepth", tag.getParameter("dp::specularreflectdepth"), "integer");
				str += paramLine("specularrefractdepth", tag.getParameter("dp::specularrefractdepth"), "integer");
				
				if (tag.getParameter("dp::usereject")) {
					str += paramLine("diffusereflectreject", tag.getParameter("dp::diffusereflectreject"), "bool");
					if (tag.getParameter("dp::diffusereflectreject")) {
						str += paramLine("diffusereflectreject_threshold", tag.getParameter("dp::diffusereflectreject_threshold"), "float");
					}
					str += paramLine("diffuserefractreject", tag.getParameter("dp::diffuserefractreject"), "bool");
					if (tag.getParameter("dp::diffuserefractreject")) {
						str += paramLine("diffuserefractreject_threshold", tag.getParameter("dp::diffuserefractreject_threshold"), "float");
					}
					str += paramLine("glossyreflectreject", tag.getParameter("dp::glossyreflectreject"), "bool");
					if (tag.getParameter("dp::glossyreflectreject")) {
						str += paramLine("glossyreflectreject_threshold", tag.getParameter("dp::glossyreflectreject_threshold"), "float");
					}
					str += paramLine("glossyrefractreject", tag.getParameter("dp::glossyrefractreject"), "bool");
					if (tag.getParameter("dp::glossyrefractreject")) {
						str += paramLine("glossyrefractreject_threshold", tag.getParameter("dp::glossyrefractreject_threshold"), "float");
					}
				}
				break;
		}
		
		return str + "\n\n";
	} else {
		return '';
	}
}

// LuxTag Volume Integrator.js
var volume_integrator_types = [ 'emission', 'single' ];

function volumeIntegratorLine( tool, cam) {
	var tag = tagForScriptName(cam, "LuxTag Volume Integrator.js");
	
	if (tag) {
		var integ = volume_integrator_types[ parseInt(tag.getParameter("volume integrator type")) ];
		var str = 'VolumeIntegrator "'+integ+'"';
		
		str += paramLine("stepsize", tag.getParameter("stepsize"), 'float');
		
		return str + "\n\n";
	} else {
		return '';
	}
}

// LuxTag Accelerator.js
var accel_types = ["none", "tabreckdtree", "grid", "bvh", "qbvh"];

function acceleratorLine(tool, cam) {
	var tag = tagForScriptName(cam, "LuxTag Accelerator.js");
	
	if (tag) {
		var accel = accel_types[ parseInt(tag.getParameter("accelerator type")) ];
		var str = 'Accelerator "'+accel+'"';
		
		switch (accel) {
			case 'tabreckdtree':
			case 'unsafekdtree':
				str += paramLine("itersectcost", tag.getParameter("intersectcost"), 'integer');
				str += paramLine("traversalcost", tag.getParameter("traversalcost"), 'integer');
				str += paramLine("emptybonus", tag.getParameter("emptybonus"), 'float');
				str += paramLine("maxprims", tag.getParameter("maxprims"), 'integer');
				str += paramLine("maxdepth", tag.getParameter("maxdepth"), 'integer');
				break;
			case 'grid':
				str += paramLine("refineimmediately", tag.getParameter("refineimmediately"), 'bool');
				break;
			case 'qbvh':
				str += paramLine("maxprimsperleaf", tag.getParameter("maxprimsperleaf"), 'integer');
				break;
			case 'none':
				break;
		}
		return str + "\n\n";
	} else {
		return '';
	}
}

function lightLine( tool, light, iter, portalshape ) {
	var str = "AttributeBegin # " + light.getParameter("name") + "\n";
	
	var mat_light = light.obj2WorldMatrix();
	var mat = final_mat.multiply( mat_light );
	
	if (light.type() == 55) { // sky light
		var tag = tagForScriptName( light, "LuxTag Sunsky.js");
		
		// lightgroup check
		if (light.owner().type() != 6 ) { // not Root object.
			str += "\tLightGroup \""+light.owner().getParameter("name").replace('"','')+"\"\n";
		}
		var light_source = "\tLightSource";
		//
		if (light.getParameter("geometry")) {
			light_source += ' "sunsky"';
		} else {
			light_source += ' "sun"';
		}
		var sundir = sunDir( light );
		//print('sundir:'+sundir);
		
		light_source += paramLine('nsamples', light.getParameter("samples"), 'integer') + 
				 paramLine('turbidity', light.getParameter("turbidity"), 'float') + 
				 paramLine('sundir', sundir, 'vector') + 
				 paramLine('gain', light.getParameter("intensity"), 'float');
		
		if (tag) light_source += paramLine('relsize', tag.getParameter("relsize"), 'float');
		
		str += light_source;
		if (portalshape != '') {
			/* these line any more need.
			var mat_light_inv = mat_light.inverse();
			var mat_portal = mat.multiply( mat_light_inv );
			str += "\n\tTransform ["+mat_portal+"]\n";
			*/
			str += portalshape;
		}
		
	} else {
		str += "\tTransform ["+mat.toString()+"]\n";
		
		// lightgroup check
		if (light.owner().type() != 6 ) { // not Root object.
			str += "\tLightGroup \""+light.owner().getParameter("name").replace('"','')+"\"\n";
		}
		//
		
		var intensity = light.getParameter("intensity");
		var color = light.getParameter("color"); //.multiply(intensity);
		
		var from = new Vec3D(0, 0, 0);
		var axis = parseInt(light.getParameter("axis"));
		switch(axis) {
			case 0: // -X
				var to = new Vec3D(-1,  0,  0);
				var createVec = function( a, b ) { return new Vec3D(0, b, a) };
				break;
			case 1: // -Y
				var to = new Vec3D( 0, -1,  0);
				var createVec = function( a, b ) { return new Vec3D(b, 0, a) };
				break;
			case 2: // -Z
				var to = new Vec3D( 0,  0, -1);
				var createVec = function( a, b ) { return new Vec3D(a, b, 0) };
				break;
		}
		var nsamples = light.getParameter("samples");
		
		switch(parseInt(light.getParameter("lightType"))) {
			case 0: // ambient -> infinite
				str += "\tLightSource";
				str += ' "infinite"' + paramLine('L',color,'color') + paramLine('nsamples', nsamples, 'integer') +
											  paramLine('gain', intensity, 'float');
				break;
			case 1: // area
				var w = light.getParameter("width");
				var h = light.getParameter("height");
				
				if (axis == 0) { // -X type
					var vec3 = createVec( w/2,  h/2);
					var vec2 = createVec( w/2, -h/2);
					var vec1 = createVec(-w/2, -h/2);
					var vec0 = createVec(-w/2,  h/2);
				} else {
					var vec0 = createVec( w/2,  h/2);
					var vec1 = createVec( w/2, -h/2);
					var vec2 = createVec(-w/2, -h/2);
					var vec3 = createVec(-w/2,  h/2);
				}
				str += "\tAreaLightSource";
				str += ' "area"' + paramLine('L', color, 'color') + paramLine('nsamples', nsamples, 'integer') +
										 paramLine('gain', intensity, 'float');
				str += "\n\tShape \"trianglemesh\" \"integer indices\" [\n";
				str += "\t\t0 1 2\n";
				str += "\t\t0 2 3\n";
				str += "\t] \"point P\" [\n";
				str += "\t\t" + vec0 + "\n";
				str += "\t\t" + vec1 + "\n";
				str += "\t\t" + vec2 + "\n";
				str += "\t\t" + vec3 + "\n";
				str += "\t]";
				break;
			case 2: // distant
				str += "\tLightSource";
				str += ' "distant"' + paramLine('L',color,'color') +
							paramLine('from', from, 'point') + paramLine('to', to, 'point') +
							paramLine('gain', intensity, 'float');
				break;
			case 3: // point or exporter's default
				str += "\tLightSource";
				str += ' "point"' + paramLine('L',color,'color') + paramLine('from', new Vec3D(0, 0, 0), 'point') +
										 paramLine('gain', intensity, 'float');
				break;
			case 4: // spot
				str += "\tLightSource";
				var coneangle = light.getParameter("cutOffAngle");
				//var conedeltaangle = light.getParameter("cutOffAttenuation"); // 0.6 anymore use this as float -> texture? how do I implement this?
				str += ' "spot"' + paramLine('L',color,'color') + paramLine('coneangle', coneangle, 'float') + 
								paramLine('gain', intensity, 'float') + paramLine('from', from, 'point') + paramLine('to', to, 'point');
				break;
		}
	}
	str += "\nAttributeEnd\n";
	
	return str;
}

function testSundir( tool ) {
	var light = tool.document().selectedObject();
	
	if (light.type() == 55) {
		var sundir = sunDir(light);
		//print('sundir:'+sundir);
		
		var childCount = light.childCount();
		if (childCount > 0) {
			light.childAtIndex(0).setParameter("position", sundir)
		}
	}
}

function decompressNormal( n, theta, phi)
{
	/*
	n.x = Math.sin(theta) * Math.cos(phi);
	n.y = Math.cos(theta);
	n.z = Math.sin(theta) * Math.sin(phi);
	*/
	n.x = -1 * (Math.sin(theta) * Math.sin(phi));
	n.y = Math.cos(theta);
	n.z = Math.sin(theta) * Math.cos(phi);
}

function sunDir( light ) {
	var sundir = new Vec3D( 0, 0, 0); // z-up;
	
	/* require sundir vector calculation */
	// Martin helped these code. special thanks :D.
	
	//var DEG2RAD = 0.01745329251994329576923690768488;
	var DEG2RAD = Math.PI / 180;
	
	var skyLightDate = light.getParameter("date");
	var skyLightLatitude = light.getParameter("latitude");
	
	var timeOfDay = (skyLightDate % 1440) / 60.0;
	var julianDay = Math.floor(skyLightDate/1440.0);
	var solarTime = timeOfDay + (0.170*Math.sin(4.0*Math.PI*(julianDay - 80)/373) - 0.129*Math.sin(2.0*Math.PI*(julianDay - 8)/355));
	var solarDeclination = (0.4093*Math.sin(2.0*Math.PI*(julianDay - 81.0)/368.0));
	var solarAltitude = Math.asin(Math.sin(DEG2RAD*skyLightLatitude) * Math.sin(solarDeclination) - Math.cos(DEG2RAD*skyLightLatitude) * Math.cos(solarDeclination) * Math.cos(Math.PI*solarTime/12));
	
	var opp = -Math.cos(solarDeclination) * Math.sin(Math.PI*solarTime/12);
	var adj = -(Math.cos(DEG2RAD*skyLightLatitude) * Math.sin(solarDeclination) + Math.sin(DEG2RAD*skyLightLatitude) * Math.cos(solarDeclination) *  Math.cos(Math.PI*solarTime/12));
	var solarAzimuth = Math.atan2(opp,adj);

	var phiS = solarAzimuth;
	var thetaS = Math.PI / 2.0 - solarAltitude;
	
	decompressNormal(sundir, thetaS, phiS + Math.PI*2);
	
	var light_mat = light.obj2WorldMatrix();
	
	return final_mat.multiply(light_mat.multiply(sundir));
	//return sundir;
}

// LuxTag Volume.js
var volume_types = ["exponential","homogenous","cloud"];

function writeVolumeLine( tool, obj, iter, file) {
	var tag = tagForScriptName(obj, "LuxTag Volume.js");
	
	var core = obj.modCore();
	var vertexCount = core.vertexCount();
	var i;
	var minVec = undefined;
	var maxVec = undefined;
	
	for (i = 0;i < vertexCount;i++) {
		var vec = core.vertex(i);
		if (minVec == undefined) {
			minVec = vec.copy();
		} else {
			minVec.x = (vec.x < minVec.x)? vec.x : minVec.x;
			minVec.y = (vec.y < minVec.y)? vec.y : minVec.y;
			minVec.z = (vec.z < minVec.z)? vec.z : minVec.z;
		}
		if (maxVec == undefined) {
			maxVec = vec.copy();
		} else {
			maxVec.x = (vec.x > maxVec.x)? vec.x : maxVec.x;
			maxVec.y = (vec.y > maxVec.y)? vec.y : maxVec.y;
			maxVec.z = (vec.z > maxVec.z)? vec.z : maxVec.z;
		}
	}
	
	var mat = final_mat.multiply( obj.obj2WorldMatrix() );
	
	var volume_type = volume_types[ parseInt( tag.getParameter("volume type") ) ];
	
	file.writeln("AttributeBegin");
	file.writeln("\tTransform [" + mat + "]");
	file.write("\tVolume \""+volume_type+"\"");
	
	file.write( paramLine("sigma_a", new Vec4D( tag.getParameter("absorption R"), tag.getParameter("absorption G"), tag.getParameter("absorption B"), 1.0 ), "color") );
	file.write( paramLine("sigma_s", new Vec4D( tag.getParameter("scattering R"), tag.getParameter("scattering G"), tag.getParameter("scattering B"), 1.0 ), "color") );
	file.write( paramLine("Le", new Vec4D( tag.getParameter("emission R"), tag.getParameter("emission G"), tag.getParameter("emission B"), 1.0 ), "color") );
	file.write( paramLine("g", tag.getParameter("asymmetry g"), "float") );
	
	file.write( paramLine("p0", minVec, "point") + paramLine("p1", maxVec, "point") );
	
	if (volume_type == "exponential") {
		file.write( paramLine("a", tag.getParameter("form a/scale"), "float") + paramLine("b", tag.getParameter("form b/falloff"), "float") + 
					paramLine("updir", new Vec3D( tag.getParameter("updir x"), tag.getParameter("updir y"), tag.getParameter("updir z")), "vector"));
	} else if (volume_type == "cloud") {
		file.write( paramLine("radius", tag.getParameter("radius"), "float") +
					paramLine("noisescale", tag.getParameter("noisescale"), "float") +
					paramLine("turbulence", tag.getParameter("turbulence"), "float") +
					paramLine("noiseoffset", tag.getParameter("noiseoffset"), "float") +
					paramLine("octaves", tag.getParameter("octaves"), "integer") +
					paramLine("omega", tag.getParameter("omega"), "float") +
					paramLine("sharpness", tag.getParameter("sharpness"), "float") +
					paramLine("variability", tag.getParameter("variability"), "float") +
					paramLine("baseflatness", tag.getParameter("baseflatness"), "float") +
					paramLine("spheres", tag.getParameter("spheres"), "integer") +
					paramLine("spheresize", tag.getParameter("spheresize"), "float")
				);
	}
	file.write("\n");
	file.writeln("AttributeEnd\n");
}

// LuxTag Material.js
// substrate -> glossy
var material_types = ["carpaint", "glass", "matte", "mattetranslucent", "metal", "mirror", "roughglass", "shinymetal", "glossy", "mix", "null", "auto"];
var solidcolor_types = ["light"];

// sub option
var carpaint_names = ["ford f8", "polaris silber", "opel titan", "bmw339", "2k acrylack", "white", "blue", "blue matte", "edit after exporting.."];
var metal_names = ['amorphous carbon', 'silver', 'gold', 'copper', 'aluminium'];
// IOR cauchy
var cauchybnames = ["01 - Fused silica glass", "02 - Borosilicate glass BK7", "03 - Hard crown glass K5", "04 - Barium crown glass BaK4", "05 - Barium flint glass BaF10", "06 - Dense flint glass SF10", "07 - use cauchybnames channel"  ]
var cauchybvals = [ 0.00354, 0.00420, 0.00459, 0.00531, 0.00743, 0.01342, -1 ]
// clipmap
var clip_maps = ['none','colorTex','specTex','reflTex','transTex'];
var clip_labels = ['none', 'ColorMap', 'SpecMap', 'ReflMap', 'TransMap'];
// channels
var channel_color = ['none', 'colorColor', 'specularColor', 'emissionColor', 'none', 'reflColor', 'transColor'];
var channel_map = ['none', 'colorTex', 'specTex', 'none', 'bumpTex', 'reflTex', 'transTex'];
var channel_mix = ['none', 'colorMix', 'specMix', 'none', 'none', 'reflMix', 'transMix'];
var channel_intensity = ['none', 'none', 'none', 'none', 'bumpIntensity', 'reflIntensity', 'transIntensity'];
var channel_labels = ['none', 'ColorChannel', 'SpecularChannel', 'EmissiveChannel', 'BumpChannel', 'ReflChannel', 'TransChannel'];
// fake tag object
function materialTag() {

}
materialTag.prototype.getParameter = function ( key ) {
	var val;
	switch(key) {
		case 'use alter ior':
			val = false; break;
		case 'alter ior':
			val = 1.0; break;
		case 'use film':
			val = 0; break;
		case 'film':
			val = 200; break;
		case 'film ior':
			val = 1.5; break;
		case 'architectural':
			val = 0; break;
		case 'chromadisp':
			val = 0; break;
		case 'cauchyb': // defualt is -1
			val = "6"; break;
		case 'carpaint name':
			val = "0"; break;
		// subdiv
		case 'subdiv levels':
			val = 0; break;
		// channels
		case 'bumpmap channel':
			val = "0"; break;
		case 'emission channel':
			val = "0"; break;
		case 'displacementmap channel':
			val = "0"; break;
		case 'Kd channel':
			val = '1'; break;
		case 'Kr channel':
			val = '5'; break;
		case 'Kt channel':
			val = '6'; break;
		case 'Ks channel':
			val = '5'; break;
		case 'uroughness channel':
		case 'vroughness channel':
			val = '2'; break;
		//
		case 'material type':
			val = '12'; break;
		case 'clipmap channel':
			val = '0'; break;
		//
		case 'use imagemap gamma':
			val = false; break;
		//
		case 'power':
			val = 100; break;
		case 'efficacy':
			val = 17; break;
		case 'gain':
			val = 1; break;
	}
	//print('[tagAlt] '+key+':'+val);
	return val;
}
//
function writeMaterialLine( tool, matInfo, iter, file ) {
	
	var mat = tool.document().materialAtIndex( matInfo[1] );
	var tag = matInfo[0];
	var mat_name = matInfo[2] + '_' + mat.getParameter("name").replace('"','') + '_' + iter; // obj_name + mat_name + iterator_num;
	var shader_type = mat.getParameter("shaderType");
	
	var bumpLayer = false;
	var emissionLayer = false;
	var subdivLayer = false;
	var displacementLayer = false;
	//
	var uvScale = tag.getParameter("UVScale");
	var uvOffset = tag.getParameter("UVOffset");
	if (uvScale.x != 1 || uvScale.y != 1) {
		var scaleLine = paramLine("uscale", uvScale.x, "float") + paramLine("vscale", uvScale.y, "float");
	} else {
		var scaleLine = '';
	}
	if (uvOffset.x != 0 || uvOffset.y != 0) {
		var offsetLine = paramLine("udelta", uvOffset.x, "float") + paramLine("vdelta", uvOffset.y, "float");
	} else {
		var offsetLine = '';
	}
	
	var clipMap = 'none'; // default is "none";
	switch (shader_type) {
		case 'Material': // type:Material
			
			var tagAlt = tagForMaterialInfo( tag.owner(), tag.getParameter("shadeSelection"), "LuxTag Material.js" );
			if (tagAlt == false) {
				tagAlt = new materialTag();
			}
			var gamma = (tagAlt.getParameter("use imagemap gamma"))? tagAlt.getParameter("imagemap gamma") : gGamma;
			//
			var matType = '';
			matType = material_types[ parseInt(tagAlt.getParameter("material type")) ];
			clipMap = clip_maps[ parseInt(tagAlt.getParameter("clipmap channel")) ];
			
			if (matType == 'auto') {
				if (transIntensity > 0) { // use glass
					matType = 'glass';
				} else if (reflIntensity > 0) { // use shiny glass
					matType = 'shinymetal';
				} else {
					matType = 'matte';
				}
			}
			// bump texture
			if (parseInt(tagAlt.getParameter("bumpmap channel"))) {
				file.writeln( channelLine(mat, 'float', parseInt(tagAlt.getParameter("bumpmap channel")), mat_name+".bumpmap", scaleLine, offsetLine, gamma) );
				bumpLayer = true;
			}
			// emission
			if (parseInt(tagAlt.getParameter("emission channel"))) {
				file.writeln( channelLine(mat, 'color', parseInt(tagAlt.getParameter("emission channel")), mat_name+".emission", scaleLine, offsetLine, gamma) );
				emissionLayer = "emission";
			}
			// subdiv
			if (tagAlt.getParameter("subdiv levels") > 0) {
				subdivLayer = [ tagAlt.getParameter("subdiv levels"), tagAlt.getParameter("subdiv sharpbound"), tagAlt.getParameter("subdiv nsmooth")];
				//print(subdivLayer);
			}
			// displacement
			if (parseInt(tagAlt.getParameter("displacementmap channel"))) {
				file.writeln( channelLine(mat, 'float', parseInt(tagAlt.getParameter("displacementmap channel")), mat_name+".displacementmap", scaleLine, offsetLine, gamma) );
				displacementLayer = [ mat_name+".displacementmap", tagAlt.getParameter("disp amount"), tagAlt.getParameter("disp offset") ];
			}
			// mat_cache
			var line_cache = '';
			//
			switch (matType) {
				case 'carpaint':
					var crp_name = carpaint_names[ parseInt(tagAlt.getParameter("carpaint name")) ];
					//
					file.write('MakeNamedMaterial "'+mat_name+'"'+paramLine('type', "carpaint", "string"));
					if (crp_name == 'edit after exporting') {
						file.write( paramLine('Kd', '', 'texture') + 
							paramLine('Ks1', '', 'texture') + paramLine('Ks2', '', 'texture') + paramLine('Ks3', '', 'texture') + 
							paramLine('R1', '', 'texture') + paramLine('R2', '', 'texture') + paramLine('R3', '', 'texture') + 
							paramLine('M1', '', 'texture') + paramLine('M2', '', 'texture') + paramLine('M3', '', 'texture'));
					} else {
						file.write( paramLine('name', crp_name, 'string') );
					}
					break;
				case 'glass':
					var ior = (tagAlt.getParameter("use alter ior"))? tagAlt.getParameter("alter ior") : mat.getParameter("transEta");
					var use_film = tagAlt.getParameter("use film");
					var film = tagAlt.getParameter("film");
					var film_ior = tagAlt.getParameter("film ior");
					var architectural = tagAlt.getParameter("architectural");
					var chromadisp = tagAlt.getParameter("chromadisp");
					var cauchyb = cauchybvals[ parseInt(tagAlt.getParameter("cauchyb")) ];
					//
					if (cauchyb < 0) {
						var cauchyb_chn = parseInt(tagAlt.getParameter("cauchyb channel"));
						file.writeln( channelLine(mat, 'float', cauchyb_chn, mat_name+'.cauchyb', scaleLine, offsetLine, gamma) );
					}
					//
					file.writeln( channelLine(mat, 'color', parseInt(tagAlt.getParameter("Kr channel")), mat_name+'.Kr', scaleLine, offsetLine, gamma) );
					file.writeln( channelLine(mat, 'color', parseInt(tagAlt.getParameter("Kt channel")), mat_name+'.Kt', scaleLine, offsetLine, gamma) );
					
					file.write('MakeNamedMaterial "'+mat_name+'"' + paramLine('type', "glass", 'string'));
					file.write( paramLine('Kr', mat_name + '.Kr', 'texture') );
					file.write( paramLine('Kt', mat_name + '.Kt', 'texture') );
					file.write( paramLine('index', ior, 'float') );
					file.write( paramLine('architectural', architectural, 'bool') );
					if (architectural == false) {
						if (chromadisp) {
							if (cauchyb < 0) {
								file.write( paramLine('cauchyb', mat_name + '.cauchyb', 'texture') );
							} else {
								file.write( paramLine('cauchyb', cauchyb, 'float') );
							}
						}
						if (use_film) {
							file.write( paramLine('film', film, 'float') );
							file.write( paramLine('filmindex', film_ior, 'float') );
						}
					}
					break;
				case 'glossy': // used be 'substrate' or 'plastic'
					var ur_chn = parseInt(tagAlt.getParameter("uroughness channel")); // : mat.getParameter("reflAngle") / 30;
					var vr_chn = parseInt(tagAlt.getParameter("vroughness channel")); // : mat.getParameter("reflAngle") / 30;
					file.writeln( channelLine(mat, 'color', parseInt(tagAlt.getParameter("Kd channel")), mat_name+'.Kd', scaleLine, offsetLine, gamma) );
					file.writeln( channelLine(mat, 'color', parseInt(tagAlt.getParameter("Ks channel")), mat_name+'.Ks', scaleLine, offsetLine, gamma) );
					file.writeln( channelLine(mat, 'color', parseInt(tagAlt.getParameter("Ka channel")), mat_name+'.Ka', scaleLine, offsetLine, gamma) );
					file.writeln( channelLine(mat, 'float', parseInt(tagAlt.getParameter("d channel")), mat_name+'.d', scaleLine, offsetLine, gamma) );
					if (ur_chn) file.writeln( channelLine(mat, 'float', ur_chn, mat_name+'.urough', scaleLine, offsetLine, gamma) );
					if (vr_chn) file.writeln( channelLine(mat, 'float', ur_chn, mat_name+'.vrough', scaleLine, offsetLine, gamma) );
					//
					file.write('MakeNamedMaterial "' + mat_name + '"' + paramLine('type', "glossy", 'string'));
					file.write( paramLine('Kd', mat_name + '.Kd', 'texture') );
					file.write( paramLine('Ks', mat_name + '.Ks', 'texture') );
					file.write( paramLine('Ka', mat_name + '.Ka', 'texture') );
					file.write( paramLine('d' , mat_name + '.d' , 'texture') );
					
					if (ur_chn) file.write( paramLine('uroughness', mat_name + '.urough', 'texture') );
					else file.write( paramLine('uroughness', mat.getParameter("reflAngle") / 30, 'float'));
					
					if (vr_chn) file.write( paramLine('vroughness', mat_name + '.vrough', 'texture') );
					else file.write( paramLine('vroughness', mat.getParameter("reflAngle") / 30, 'float'));
					break;
				case 'matte':
					var sigma = tagAlt.getParameter("sigma channel");
					var sigma_val = tagAlt.getParameter("sigma value");
					// color texture
					file.writeln( channelLine(mat, 'color', parseInt(tagAlt.getParameter("Kd channel")), mat_name+'.Kd', scaleLine, offsetLine, gamma) );
					file.writeln( channelLine(mat, 'float', parseInt(tagAlt.getParameter("sigma channel")), mat_name+'.sigma', scaleLine, offsetLine, gamma, sigma_val) );
					
					//
					file.write('MakeNamedMaterial "'+mat_name+'"' + paramLine('type', "matte", 'string'));
					file.write( paramLine('Kd', mat_name + '.Kd', 'texture') );
					file.write( paramLine('sigma', mat_name + '.sigma', 'texture') );
					
					break;
				case 'mattetranslucent':
					var sigma = tagAlt.getParameter("sigma channel");
					var sigma_val = tagAlt.getParameter("sigma value");
					file.writeln( channelLine(mat, 'color', parseInt(tagAlt.getParameter("Kr channel")), mat_name+'.Kr', scaleLine, offsetLine, gamma) );
					file.writeln( channelLine(mat, 'color', parseInt(tagAlt.getParameter("Kt channel")), mat_name+'.Kt', scaleLine, offsetLine, gamma) );
					file.writeln( channelLine(mat, 'float', parseInt(tagAlt.getParameter("sigma channel")), mat_name+'.sigma', scaleLine, offsetLine, gamma, sigma_val) );
					//
					file.write('MakeNamedMaterial "'+mat_name+'"' + paramLine('type', "mattetranslucent", 'string'));
					file.write( paramLine('Kr', mat_name + '.Kr', 'texture') )+
					file.write( paramLine('Kt', mat_name + '.Kt', 'texture') );
					file.write( paramLine('sigma', mat_name + '.sigma', 'texture') );
					break;
				case 'metal':
					var metal_name = metal_names[ parseInt(tagAlt.getParameter('metal name')) ];
					var ur_chn = parseInt(tagAlt.getParameter("uroughness channel")); // : mat.getParameter("reflAngle") / 30;
					var vr_chn = parseInt(tagAlt.getParameter("vroughness channel")); // : mat.getParameter("reflAngle") / 30;
					if (ur_chn) file.writeln( channelLine(mat, 'float', ur_chn, mat_name+'.urough', scaleLine, offsetLine, gamma) );
					if (vr_chn) file.writeln( channelLine(mat, 'float', ur_chn, mat_name+'.vrough', scaleLine, offsetLine, gamma) );
					//
					file.write('MakeNamedMaterial "'+mat_name+'"' + paramLine('type', "metal", 'string'));
					file.write( paramLine('name', metal_name, 'string') );
					
					if (ur_chn) file.write( paramLine('uroughness', mat_name + '.urough', 'texture') );
					else file.write( paramLine('uroughness', mat.getParameter("reflAngle") / 30, 'float'));
					
					if (vr_chn) file.write( paramLine('vroughness', mat_name + '.vrough', 'texture') );
					else file.write( paramLine('vroughness', mat.getParameter("reflAngle") / 30, 'float'));
					break;
				case 'mirror':
					var film = tagAlt.getParameter("film");
					var film_ior = tagAlt.getParameter("film ior");
					file.writeln( channelLine(mat, 'color', parseInt(tagAlt.getParameter("Kr channel")), mat_name+'.Kr', scaleLine, offsetLine, gamma) );
					//
					file.write('MakeNamedMaterial "'+mat_name+'"' + paramLine('type', "mirror", 'string'));
					file.write( paramLine('Kr', mat_name + '.Kr', 'texture') );
					if (tagAlt.getParameter("use film")) {
						file.write( paramLine('film', film, 'float') + paramLine('filmindex', film_ior, 'float') );
					}
					break;
				case 'roughglass':
					var ior = (tagAlt.getParameter("use alter ior"))? tagAlt.getParameter("alter ior") : mat.getParameter("transEta");
					var chromadisp = tagAlt.getParameter("chromadisp");
					var cauchyb = cauchybvals[ parseInt(tagAlt.getParameter("cauchyb")) ];
					
					file.writeln( channelLine(mat, 'color', parseInt(tagAlt.getParameter("Kr channel")), mat_name+'.Kr', scaleLine, offsetLine, gamma) );
					file.writeln( channelLine(mat, 'color', parseInt(tagAlt.getParameter("Kt channel")), mat_name+'.Kt', scaleLine, offsetLine, gamma) );
					var ur_chn = parseInt(tagAlt.getParameter("uroughness channel")); // : mat.getParameter("reflAngle") / 30;
					var vr_chn = parseInt(tagAlt.getParameter("vroughness channel")); // : mat.getParameter("reflAngle") / 30;
					if (ur_chn) file.writeln( channelLine(mat, 'float', ur_chn, mat_name+'.urough', scaleLine, offsetLine, gamma) );
					if (vr_chn) file.writeln( channelLine(mat, 'float', ur_chn, mat_name+'.vrough', scaleLine, offsetLine, gamma) );
					if (cauchyb < 0) file.writeln( channelLine(mat, 'float', parseInt(tagAlt.getParameter("cauchyb channel")), mat_name+'.cauchyb', scaleLine, offsetLine, gamma) );
					//
					file.write('MakeNamedMaterial "'+mat_name+'"' + paramLine('type', "roughglass", 'string'));
					file.write( paramLine('Kr', mat_name + '.Kr', 'texture') + paramLine('Kt', mat_name + '.Kt', 'texture') );
					file.write( paramLine('index', ior, 'float') );
					if (chromadisp) {
						if (cauchyb < 0) {
							file.write( paramLine('cauchyb', mat_name + '.cauchyb', 'texture') );
						} else {
							file.write( paramLine('cauchyb', cauchyb, "float") );
						}
					}
					if (ur_chn) file.write( paramLine('uroughness', mat_name + '.urough', 'texture') );
					else file.write( paramLine('uroughness', mat.getParameter("reflAngle") / 30, 'float'));
					
					if (vr_chn) file.write( paramLine('vroughness', mat_name + '.vrough', 'texture') );
					else file.write( paramLine('vroughness', mat.getParameter("reflAngle") / 30, 'float'));
					break;
				case 'shinymetal':
					file.writeln( channelLine(mat, 'color', parseInt(tagAlt.getParameter("Kr channel")), mat_name+'.Kr', scaleLine, offsetLine, gamma) );
					file.writeln( channelLine(mat, 'color', parseInt(tagAlt.getParameter("Kr channel")), mat_name+'.Ks', scaleLine, offsetLine, gamma) );
					var ur_chn = parseInt(tagAlt.getParameter("uroughness channel")); // : mat.getParameter("reflAngle") / 30;
					var vr_chn = parseInt(tagAlt.getParameter("vroughness channel")); // : mat.getParameter("reflAngle") / 30;
					if (ur_chn) file.writeln( channelLine(mat, 'float', ur_chn, mat_name+'.urough', scaleLine, offsetLine, gamma) );
					if (vr_chn) file.writeln( channelLine(mat, 'float', ur_chn, mat_name+'.vrough', scaleLine, offsetLine, gamma) );
					
					var film = tagAlt.getParameter("film");
					var film_ior = tagAlt.getParameter("film ior");
					//
					file.write('MakeNamedMaterial "'+mat_name+'"' + paramLine('type', "shinymetal", 'string'));
					file.write( paramLine('Kr', mat_name + '.Kr', 'texture') + paramLine('Ks', mat_name + '.Ks', 'texture') );
					if (ur_chn) file.write( paramLine('uroughness', mat_name + '.urough', 'texture') );
					else file.write( paramLine('uroughness', mat.getParameter("reflAngle") / 30, 'float'));
					
					if (vr_chn) file.write( paramLine('vroughness', mat_name + '.vrough', 'texture') );
					else file.write( paramLine('vroughness', mat.getParameter("reflAngle") / 30, 'float'));
					
					if (tagAlt.getParameter("use film")) {
						file.write( paramLine('film', film, 'float') + paramLine('filmindex', film_ior, 'float') );
					}
					break;
				case 'mix':
					file.write('MakeNamedMaterial "'+mat_name+'"'+paramLine('type', 'mix', 'string'));
					file.write( paramLine('namedmaterial1', '<<INPUT REQUIRED>>', 'string') );
					file.write( paramLine('namedmaterial2', '<<INPUT REQUIRED>>', 'string') );
					file.write( paramLine('amount', tagAlt.getParameter('mix::amount'), 'float') );
					break;
				case 'null':
					file.write('MakeNamedMaterial "'+mat_name+'"'+paramLine('type', 'null', 'string'));
					break;
				default:
					file.writeln( channelLine(mat, 'color', parseInt(tagAlt.getParameter("Kd channel")), mat_name+'.Kd', scaleLine, offsetLine, gamma) );
					
					file.write('MakeNamedMaterial "'+mat_name+'"'+paramLine('type', "matte", 'string'));
					file.write( paramLine('Kd', mat_name + '.Kd', 'texture') );
				}
			break;
		case 'SolidColor':
			var tagAlt = tagForMaterialInfo( tag.owner(), tag.getParameter("shadeSelection"), "LuxTag SolidColor.js" );
			var gamma = 2.2; //(tagAlt)? tagAlt.getParameter("imagemap gamma") : 2.2;
			
			var colorColor = mat.getParameter("colorColor");
			var colorTex = mat.getParameter("colorTex");
			
			file.writeln( textureLine('color', colorTex, colorColor, 1.0, mat_name+".emission", scaleLine, offsetLine, gamma) );
			
			var solidcolor_type = (tagAlt)? solidcolor_types[ parseInt(tagAlt.getParameter("solidcolor type")) ] : solidcolor_types[ 0 ];
			emissionLayer = solidcolor_type;
			
			break;
		default:
			//
			file.write('MakeNamedMaterial "' + mat_name + '"' + paramLine('type', "matte", 'string') + 
													paramLine("Kd", new Vec4D(0.8, 0.8, 0.8, 1.0), 'color'));
	}
	
	if (bumpLayer)	file.write( paramLine('bumpmap', mat_name + '.BMP', 'texture' ) );
	
	if (clipMap != 'none') {
		var clipTex = mat.getParameter(clipMap);
		if (clipTex != 'none') {
			file.writeln( "\n"+textureLine('float', clipTex, null, 1.0, mat_name+".amount", scaleLine, offsetLine, gamma) );
			file.writeln( "MakeNamedMaterial \""+mat_name+".null\"" + paramLine("type", "null", "string") );
			file.write( "MakeNamedMaterial \""+mat_name+".clipped\"" + paramLine("type", "mix", "string") );
			file.write( paramLine("amount", mat_name+".amount", "texture") );
			file.write( paramLine("namedmaterial1", mat_name + ".null", "string") + paramLine("namedmaterial2", mat_name, "string") );
		}
		matInfo.push( mat_name + ".clipped" );
	} else {
		matInfo.push( mat_name ); // material name cache
	}
	file.writeln("\n");
	
	matInfo.push( emissionLayer ); // light material
	matInfo.push( subdivLayer );
	matInfo.push( displacementLayer ); // disp
}

function channelLine(mat, valType, chnNum, matName, scaleLine, offsetLine, gamma, scaleVal) {
	//print('channelLine:'+mat);
	var paramTex = channel_map[ chnNum ];
	var texVal = (paramTex != 'none')? mat.getParameter( channel_map[ chnNum ] ) : 'none';
	var paramColor = channel_color[ chnNum ];
	var colorVal = (paramColor != 'none')? mat.getParameter( channel_color[ chnNum ] ) : new Vec4D( 0, 0, 0, 1 );
	var paramMix = channel_mix[ chnNum ];
	var valMix = (paramMix != 'none')? mat.getParameter( paramMix ) : 1.0;
	if (scaleVal) {
		var valInt = scaleVal;
	} else {
		var paramInt = channel_intensity[ chnNum ];
		var valInt = (paramInt != 'none')? mat.getParameter( paramInt ) : 'none';
	}
	if (valInt == 'none') {
		return textureLine(valType, texVal, colorVal, valMix, matName, scaleLine, offsetLine, gamma);
	} else {
		return scaledTextureLine(valType, texVal, colorVal, valMix, matName, scaleLine, offsetLine, gamma, valInt);
	}
}

function textureLine(valType, texVal, colorVal, valMix, matName, scaleLine, offsetLine, gamma) {
	var str = '';
	if (texVal != 'none') {
		if (valMix == 1.0) {
			str += imageMapLine( texVal, matName, valType, gamma) + scaleLine + offsetLine;
		} else {
			str += imageMapLine( texVal, matName+'.Col', valType, gamma) + scaleLine + offsetLine + "\n";
			str += 'Texture "' + matName + '" "'+valType+'" "mix" "texture tex1" ["' + matName+'.Col"] "color tex2" [' + colorVal + ']' + paramLine("amount", valMix, "float");
		}
	} else {
		str += 'Texture "'+matName+'" "'+valType+'" "constant"'+paramLine('value', colorVal, 'color');
	}
	return str;
}

function scaledTextureLine(valType, texVal, colorVal, valMix, matName, scaleLine, offsetLine, gamma, scale) {
	if (scale == 1.0) {
		return textureLine(valType, texVal, colorVal, valMix, matName, scaleLine, offsetLine, gamma);
	} else {
		var str = '';
		if (texVal != 'none') {
			if (valMix == 1.0) {
				str += imageMapLine( texVal, matName+".T1", valType, gamma) + scaleLine + offsetLine + "\n";
				str += 'Texture "' + matName + '" "'+valType+'" "scale" "texture tex1" ["' + matName + '.T1"] "color tex2" [' + scale + ' ' + scale + ' ' + scale + ']';
			} else {
				str += imageMapLine( texVal, matName+'.Col', 'color', gamma) + scaleLine + offsetLine + "\n";
				str += 'Texture "' + matName + '.T1" "'+valType+'" "mix" "texture tex1" ["' + matName+'.Col"] "color tex2" [' + colorVal + ']' + 
											paramLine("amount", valMix, "float") + "\n";
				str += 'Texture "' + matName + '" "'+valType+'" "scale" "texture tex1" ["' + matName+'.T1"] "color tex2" [' + scale + ' ' + scale + ' ' + scale + ']';
			}
		} else {
			str += 'Texture "'+matName+'.T1" "'+valType+'" "constant"'+paramLine('value', colorVal, 'color') + "\n";
			str += 'Texture "' + matName + '" "' + valType + '" "scale" "texture tex1" ["' + matName + '.T1"] "color tex2" [' + scale + ' ' + scale + ' ' + scale + ']';
		}
	}
	return str;
}

function imageMapLine( tex, texName, texType, gamma ) {
	var str = 'Texture "' + texName + '" "' + texType + '" "imagemap"' + 
			paramLine('filename', tex, 'string') + paramLine('gamma', gamma, 'float') + paramLine('wrap', 'repeat', 'string');
	
	return str;
}

function makeMaterialAssignmentCache( tool, obj, iter ) {
	
	var i,j;
	var matInfo = material_list[iter];
	// materialTag, linkNum, fileHundle, matName, emissionLayer
	var mat_res = [ [ null, -1, new File("/tmp/"+cache_name+"-1.dat"), 'luxCh_default', false ] ]; // no assignment
	var activePolySel = obj.getParameter("activePolySelection");
	
	var core = obj.modCore();
	var polyCount = core.polygonCount();
	
	var cache = new File("/tmp/"+cache_name+"holder.dat");
	cache.open(WRITE_MODE);
	
	for (i = 0;i < polyCount;i++) {
		cache.writeInt(-1);
	}
	
	//
	for (i = 0;i < matInfo.length;i++) {
		var tag = matInfo[i][0];
		var selnum = tag.getParameter("shadeSelection");
		var linkNum = matInfo[i][1];
		var matName = matInfo[i][3];
		var emissionLayer = matInfo[i][4];
		var subdivLayer = matInfo[i][5];
		var displacementLayer = matInfo[i][6];
		
		var mat_insert = false;
		for (j = 0;j < mat_res.length;j++) {
			if (mat_res[j][1] == linkNum) {
				mat_insert = true;
			}
		}
		if (mat_insert == false) {
			mat_res.push( [ tag, linkNum, new File("/tmp/"+cache_name+linkNum+".dat"), matName, emissionLayer, subdivLayer, displacementLayer ] );
		}
		
		if (selnum > 0) core.setActivePolygonSelection( selnum - 1 );
		
		for (j = 0;j < polyCount;j++) {
			if (selnum > 0) {
				if (core.polygonSelection(j)) {
					cache.setpos( j * intSize );
					cache.writeInt( linkNum );
				}
			} else { // all
				cache.setpos( j * intSize );
				cache.writeInt( linkNum );
			}
		}
	}
	cache.close();
	cache.open(READ_MODE);
	
	// open cache
	for (i = 0;i < mat_res.length;i++) {
		var file = mat_res[i][2];
		file.open(WRITE_MODE);
		if (file.isOpen() == false) {
			print('File IO Error Occur!');
			return;
		}
	}
	// write over info
	for (i = 0;i < polyCount;i++) {
		linkNum = cache.readInt();
		for (j = 0;j < mat_res.length;j++) {
			if (mat_res[j][1] == linkNum) {
				file = mat_res[j][2];
				file.writeInt( i );
				break;
			}
		}
	}
	// close cache;
	for (i = 0;i < mat_res.length;i++) {
		var file = mat_res[i][2];
		file.writeInt( -1 ); // write limitter
		file.close();
	}
	
	cache.close();
	OS.system('rm ' + cache.directory() + '/' + cache.lastPathComponent());
	
	obj.setParameter("activePolySelection", activePolySel);
	obj.update();
	
	return mat_res;
}

function cleanUpMaterialAssignmentCache( mat_res ) {
		
	for (i = 0;i < mat_res.length;i++) {
		//print('linkedNum:'+mat_res[i][0]);
		var file = mat_res[i][2];
		file.open(READ_MODE);
		
		var str = '';
		var val = file.readInt();
		while (val != -1) {
			str += val + ',';
			val = file.readInt();
		}
		file.close();
		
		OS.system('rm ' + file.directory() + '/' + file.lastPathComponent());
	}
}

function writeMeshLine( tool, obj, iter, file, master_file ) {
	var obj_name = obj.getParameter("name").replace('"','') + '_' + iter;
	var i,j, polySize, indices, len;
	
	var core = obj.modCore();
	
	var mat_res = makeMaterialAssignmentCache( tool, obj, iter );
	
	// light check
	var mat = final_mat.multiply( obj.obj2WorldMatrix() );
	var isLight = false
	for (i = 0;i < mat_res.length;i++) {
		if (mat_res[i][4] == 'emission' || mat_res[i][4] == 'portal' || mat_res[i][4] == 'light') {
			isLight = true;
		}
	}
	if (isLight) {
		file.writeln("AttributeBegin # " + obj_name + "\n");
		file.writeln("\tTransform ["+mat+"]");
	} else {
		file.writeln("ObjectBegin \"" + obj_name +"\"\n");
	}
	
	for (i = 0;i < mat_res.length;i++) {
		var cache = mat_res[i][2]; // file
		var tri_index = 0;
		var vert_cache = new File("/tmp/"+cache_name+"verts.dat");
		
		cache.open(READ_MODE);
		
		var val = cache.readInt();
		if (val != -1) {
			
			if (mat_res[i][4] == 'emission' || mat_res[i][4] == 'light') {
				var tag = mat_res[i][0];
				var material = tool.document().materialAtIndex( mat_res[i][1] );
				if (mat_res[i][4] == 'emission') {
					if (mat_res[i][0] == -1) { // default
						file.writeln("\tNamedMaterial \"luxCh_default\"");
					} else {
						file.writeln("\tNamedMaterial \"" + mat_res[i][3] + "\"");
					}
					var tagAlt = tagForMaterialInfo( obj, tag.getParameter("shadeSelection"), "LuxTag Material.js" );
				} else {
					var tagAlt = tagForMaterialInfo( obj, tag.getParameter("shadeSelection"), "LuxTag SolidColor.js" );
				}
				if (tagAlt == false) {
					tagAlt = new materialTag();
				}
				
				// lightgroup check
				if (obj.owner().type() != 6) {
					file.writeln("\tLightGroup \""+obj.owner().getParameter("name").replace('"','')+"\"");
				}
				//
				var emiPower = tagAlt.getParameter("power");
				var emiEfficacy = tagAlt.getParameter("efficacy");
				var emiGain = tagAlt.getParameter("gain");
				
				file.writeln("\tAreaLightSource \"area\" \"texture L\" [\"" + mat_res[i][3] + '.emission' + "\"]" + 
							paramLine('power', emiPower, 'float') +
							paramLine('efficacy', emiEfficacy, 'float') +
							paramLine('gain', emiGain, 'float'));
			} else {
				if (mat_res[i][0] == -1) { // default
					file.writeln("\tNamedMaterial \"luxCh_default\"");
				} else {
					file.writeln("\tNamedMaterial \"" + mat_res[i][3] + "\"");
				}
			}
			
			//print('mat_res[i]:'+mat_res[i]);
			
			if (mat_res[i][5] && mat_res[i][5] != false) { // subdiv
				var subdivInfo = mat_res[i][5];
				
				file.write( "\tShape" + ' "loopsubdiv" ' + paramLine("nlevels", subdivInfo[0], "integer") + 
														 paramLine("dmnormalsmooth", subdivInfo[1], "bool") + 
														 paramLine("dmsharpboundary", subdivInfo[2], "bool") );
			} else {
				file.write("\tShape \"trianglemesh\"");
			}
			//
			if (mat_res[i][6] && mat_res[i][6] != false) { // disp
				var dispInfo = mat_res[i][6];
				
				file.write( paramLine("displacementmap", dispInfo[0], "string") + paramLine("dmscale", dispInfo[1], "float") + paramLine("dmoffset", dispInfo[2], "float") );
			}
			
			file.writeln(' "integer indices" [');
			
			// vertices cache
			vert_cache.open(WRITE_MODE);
		}
		var vert_count = 0;
		while (val != -1) {
			polySize = core.polygonSize( val );
			vert_cache.writeInt(val);
			vert_cache.writeInt(polySize);
			tri_index++;
			for (j = 0;j < polySize - 2;j++) {
				indices = core.triangle( val, j );
				
				file.writeln("\t\t"+(vert_count+indices[0])+' '+(vert_count+indices[1])+' '+(vert_count+indices[2]));
			}
			vert_count += polySize;
			/*
			for (j = 0;j < polySize - 2;j++) {
				indices = core.triangle( val, j);
				
				vert_cache.writeInt(val);
				vert_cache.writeInt( indices[0] );
				vert_cache.writeInt( indices[1] );
				vert_cache.writeInt( indices[2] );
				
				file.writeln("\t\t"+(tri_index*3) + ' ' + (tri_index*3+1) + ' ' + (tri_index*3+2));
				//file.writeln("\t\t"+indices[0]+' '+indices[1]+' '+indices[2]);
				tri_index++;
			}
			*/
			val = cache.readInt();
		}
		cache.close();
		if (tri_index > 0) {
			var pi, ps;
			vert_cache.close();
			vert_cache.open(READ_MODE);
			
			file.writeln("\t] \"point P\" [");
			for (j = 0;j < tri_index;j++) {
				pi = vert_cache.readInt(); // poly index
				ps = vert_cache.readInt(); // poly size
				
				for (k = 0;k < ps;k++) {
					file.writeln("\t\t"+core.vertex( core.vertexIndex( pi, k ) ));
				}
			}
			vert_cache.seek(0, SEEK_SET);
			file.writeln("\t] \"normal N\" [");
			for (j = 0;j < tri_index;j++) {
				pi = vert_cache.readInt();
				ps = vert_cache.readInt();
				
				for (k = 0;k < ps;k++) {
					file.writeln("\t\t"+core.normal( pi, k ) );
				}
			}
			vert_cache.seek(0, SEEK_SET);
			file.writeln("\t] \"float uv\" [");
			for (j = 0;j < tri_index;j++) {
				pi = vert_cache.readInt();
				ps = vert_cache.readInt();
				
				for (k = 0;k < ps;k++) {
					file.writeln("\t\t"+core.uvCoord( pi, k ).toUV() );
				}
			}
			file.writeln("\t]\n");
			
			vert_cache.close();
			OS.system("rm "+vert_cache.directory()+'/'+vert_cache.lastPathComponent());
		}
	}
	
	if (isLight) {
		file.writeln("AttributeEnd\n");
	} else {
		file.writeln("ObjectEnd\n");
		
		master_file.writeln("AttributeBegin");
		master_file.writeln("\tTransform [" + mat + "]");
		master_file.writeln("\tObjectInstance \""+obj_name+"\"");
		master_file.writeln("AttributeEnd\n");
	}
	cleanUpMaterialAssignmentCache( mat_res );
}

function storeObjects( obj, tool ) {
	var len = obj.childCount();
	for (var i = 0;i < len;i++) {
		var child = obj.childAtIndex( i );
		//
		var childType = child.type();
		var modeTag = tagForType( child, 103 );
		if (modeTag && modeTag.getParameter("renderActive")) { // only render on
			if (child.family() == NGONFAMILY) {
				var volTag = tagForScriptName(child, "LuxTag Volume.js");
				var portalTag = tagForScriptName(child, "LuxTag Portal.js");
				if (volTag != false) {
					vol_list.push(child);
				} else if (portalTag != false) {
					portal_list.push(child);
				} else {
					var matInfo = [];
					var matTags = child.materialTags();
					for (var j = 0;j < matTags.length;j++) {
						var linkNum = matTags[j].linkedToMaterial();
						matInfo.push( [ matTags[j], linkNum, child.getParameter("name").replace('"','') ] );
					}
					material_list.push(matInfo);
					mesh_list.push(child);
				}
			} else if (child.family() == LIGHTFAMILY) {
				light_list.push(child);
			}
			
			if ( childType == SYMMETRY || childType == BOOLEAN || childType == CHAIN || childType == EXTRUDE ||
								childType == LATHE || childType == POLYPLANE || childType == SWEEP ) { // stop seek if obj is creator.
				//
			} else {
				storeObjects( child, tool );
			}
		}
	}
}

// getting tag for type
function tagForType(obj, tagType) {
	var tagCount = obj.tagCount();
	var resultTag = false;
	for (var i = 0;i < tagCount;i++) {
		var tag = obj.tagAtIndex(i);
		if (tag.type() == tagType && tag.getParameter("tagOn")) {
			resultTag = tag;
		}
	}
	return resultTag;
}

function tagsForType(obj, tagType) {
	var tagCount = obj.tagCount();
	var resultTags = [];
	for (var i = 0;i < tagCount;i++) {
		var tag = obj.tagAtIndex(i);
		if (tag.type() == tagType && tag.getParameter("tagOn")) {
			resultTags.push(tag);
		}
	}
	return resultTags;
}

function tagForScriptName(obj, scriptName) {
	var tagCount = obj.tagCount();
	var resultTag = false;
	for (var i = 0;i < tagCount;i++) {
		var tag = obj.tagAtIndex(i);
		if (tag.getParameter("scriptName") == scriptName && tag.getParameter("tagOn")) {
			resultTag = tag;
		}
	}
	return resultTag;
}

function tagForMaterialInfo(obj, selectionNum, scriptName) {
	var tagCount = obj.tagCount();
	var resultTag = false;
	for (var i = 0;i < tagCount;i++) {
		var tag = obj.tagAtIndex(i);
		if (tag.getParameter("scriptName") == scriptName && tag.getParameter("tagOn") && tag.getParameter("shade selection") == selectionNum - 1) {
			resultTag = tag;
		}
	}
	return resultTag;
}