//
//  MandelbulbApp.m
//  Mandelbulb
//
//  Created by Hiroto Tsubaki on 10/03/10.
//  Copyright 2010 Hiroto Tsubaki. All rights reserved.
//
// reference codes from 
//
// Copyright (c) 2009 Tom Beddard
// http://www.subblue.com
//
// Licensed under the MIT License:
// http://www.opensource.org/licenses/mit-license.php
//
//
// Credits and references
// ======================
// For the story behind the 3D Mandelbrot see the following page:
// http://www.skytopia.com/project/fractal/mandelbulb.html
//
// The original forum disussion with many implementation details can be found here:
// http://www.fractalforums.com/3d-fractal-generation/true-3d-mandlebrot-type-fractal/
//

#import "MandelbulbApp.h"
#import "MarchingCube.h"

#include <math.h>

#define MIN_EPSION 3e-7
#define CURRENT_MARKETING_VERSION "0.2"

//

typedef struct Vec3D {
	double x,y,z;
} Vec3D;

typedef struct Vec2D {
	double x,y;
} Vec2D;

//

float Vec3D_length ( Vec3D p ) {
	return sqrt( p.x*p.x + p.y*p.y + p.z*p.z );	
}

// global variables

float power;
int iterations;
int resolutions;

BOOL julia;
struct Vec3D julia_c;

BOOL radiolaria;
float radiolariaFactor;

float bailout;

struct Vec2D phase;

//

double DE( Vec3D z0, float min_dist ) {
	
	Vec3D c = (julia)? julia_c : z0;
	Vec3D z = z0;
	float pd = power - 1.0;
	
	// Convert z to polar coordinates
	float r = Vec3D_length(z);
	float th = atan2(z.y, z.x);
	float ph = asin( z.z / r );
	
	if ( r < min_dist ) min_dist = r;
	
	Vec3D dz;
	float ph_dz = 0.0;
	float th_dz = 0.0;
	float r_dz = 1.0;
	float powR, powRsin;
	
	int n;
	
	for (n = 0;n < iterations;n++) {
		powR = power * pow(r, pd);
		powRsin = powR * r_dz * sin(ph_dz + pd*ph);
		dz.x = powRsin * cos(th_dz + pd*th) + 1.0;
		dz.y = powRsin * sin(th_dz + pd*th);
		dz.z = powR * r_dz * cos(ph_dz + pd*ph);
		
		r_dz = Vec3D_length(dz);
		th_dz = atan2(dz.y, dz.x);
		ph_dz = acos(dz.z / r_dz);
		
		powR = pow(r, power);
		powRsin = sin(power*ph);
		z.x = powR * powRsin * cos(power*th);
		z.y = powR * powRsin * sin(power*th);
		z.z = powR * cos(power*ph);
		
		z.x += c.x;
		z.y += c.y;
		z.z += c.z;
		
		if (radiolaria && z.y > radiolariaFactor) z.y = radiolariaFactor;
		
		r = Vec3D_length(z);
		if (r < min_dist) min_dist = r;
		if (r > bailout) break;
		
		th = atan2(z.y, z.x)+phase.x;
		ph = acos(z.z / r)+phase.y;
	}
	
	return 0.5 * r * log(r)/r_dz;
}

//

@implementation MandelbulbApp

- (id) init;
{
	self = [super init];
	
	if (self == nil) {
		return nil;
	}
	
	_power = 8.0f;
	_bailout = 4.0f;
	
	_phase_x = 0.0;
	_phase_y = 0.0;
	
	_resolutions = 4;
	
	_julia = false;
	_radiolaria = false;
	
	_iterations = 4;
	
	_scale = 10;
	
	_inner = 0.5;
	_outer = 1.414213562373095;
	
	_minx = -1.414213562373095;
	_miny = -1.414213562373095;
	_minz = -1.414213562373095;

	_maxx = 1.414213562373095;
	_maxy = 1.414213562373095;
	_maxz = 1.414213562373095;

	_version = false;
	_help = false;
	
	_iso = 0.5;
	_smooth = 0;
	
	_useGCD = false;
	
	return self;
}

- (void) application: (DDCliApplication *) app
    willParseOptions: (DDGetoptLongParser *) optionsParser;
{
    [optionsParser setGetoptLongOnly: YES];
	
    DDGetoptOption optionTable[] = 
    {
        // Long         Short      Argument options
		{@"output",          'o',    DDGetoptRequiredArgument},
		{@"s_output",        's',    DDGetoptRequiredArgument},
        {@"resolutions",     'r',    DDGetoptRequiredArgument},
        {@"power",           'p',    DDGetoptRequiredArgument},
        {@"bailout",          0 ,    DDGetoptRequiredArgument},
        
		{@"phase_x",          0 ,    DDGetoptRequiredArgument},
        {@"phase_y",          0 ,    DDGetoptRequiredArgument},
		
        {@"iterations",      'i',    DDGetoptRequiredArgument},
		
        {@"inner",            0 ,    DDGetoptRequiredArgument},
        {@"outer",            0 ,    DDGetoptRequiredArgument},
		
        {@"scale",           'e',    DDGetoptRequiredArgument},

        {@"minx",              0 ,    DDGetoptRequiredArgument},
        {@"miny",              0 ,    DDGetoptRequiredArgument},
        {@"minz",              0 ,    DDGetoptRequiredArgument},
		
        {@"maxx",              0 ,    DDGetoptRequiredArgument},
        {@"maxy",              0 ,    DDGetoptRequiredArgument},
        {@"maxz",              0 ,    DDGetoptRequiredArgument},
		
        {@"julia",            0 ,    DDGetoptNoArgument},
        {@"julia_x",          0 ,    DDGetoptRequiredArgument},
        {@"julia_y",          0 ,    DDGetoptRequiredArgument},
        {@"julia_z",          0 ,    DDGetoptRequiredArgument},
        
        {@"radiolaria",       0 ,    DDGetoptNoArgument},
		{@"radiolariaFactor", 0 ,    DDGetoptRequiredArgument},
		{@"smooth",           0 ,	 DDGetoptRequiredArgument},
        {@"iso",              0 ,	 DDGetoptRequiredArgument},
		
        {@"useGCD",          'g',	 DDGetoptNoArgument},
		
		
        {@"version",          0 ,    DDGetoptNoArgument},
        {@"help",            'h',    DDGetoptNoArgument},
        {nil,                 0 ,    0},
    };
	
    [optionsParser addOptionsFromTable: optionTable];
}

- (void) printUsage: (FILE *) stream;
{
    ddfprintf(stream, @"%@: Usage [OPTIONS] <argument> [...]\n", DDCliApp);
}

- (void) printHelp;
{
    [self printUsage: stdout];
    printf("\n"
           "  -o, --output OUTPUT FILE      Output file path\n"
           "  -s, --s_output OUTPUT FILE    Surface Output file path\n"
           "  -r, --resolutions INT         Fractal resolution value\n"
           "      --smooth INT              Smooth value\n"		   
           "      --iso FLOAT               Iso value for mesh creation\n"
           "  -p, --power INT               Fractal power value\n"
           "      --bailout FLOAT           Bailout value\n"
           "      --phase_x FLOAT           phase x value\n"
           "      --phase_y FLOAT           phase y value\n"
		   "  -e  --scale FLOAT             grid scale\n"
           "  -i, --iterations INT          Fractal iterations\n"
           "      --inner FLOAT             Inner radius for no calculation\n"
           "      --outer FLOAT             Outer radius for no calculation\n"
           "      --julia                   As Julia set\n"
           "      --julia_x FLOAT           julia_c x value\n"
           "      --julia_y FLOAT           julia_c y value\n"
           "      --julia_z FLOAT           julia_c z value\n"
           "      --radiolaria              As radiolaria\n"
           "      --radiolariaFactor FLOAT  Radiolaria factor\n"
           "      --min[x|y|z] FLOAT        Calculation start value\n"
           "      --max[x|y|z] FLOAT        Calculation end value\n"
		   
           "  -g, --useGCD                  Use Grand Central Dispatch\n"
		   
           "      --version                 Display version and exit\n"
           "  -h, --help                    Display this help and exit\n"
           "\n");
}

- (void) printVersion;
{
    ddprintf(@"%@ version %s\n", DDCliApp, CURRENT_MARKETING_VERSION);
}

- (int) application: (DDCliApplication *) app
   runWithArguments: (NSArray *) arguments;
{
    if (_help)
    {
        [self printHelp];
        return EXIT_SUCCESS;
    }
    
    if (_version)
    {
        [self printVersion];
        return EXIT_SUCCESS;
    }
	
	if (_useGCD) {
		//NSLog(@"using Grand Central Dispatch code.");
	}
	
	power = _power;
	iterations = _iterations;
	
	resolutions = _resolutions;
	
	double dx = _maxx - _minx;
	double dy = _maxy - _miny;
	double dz = _maxz - _minz;
	
	double d_max = (dx > dy)?
						(dx > dz)? dx : dz :
						(dy > dz)? dy : dz ;
	double d = d_max / resolutions;
	
	int resx = ceil( dx / d );
	int resy = ceil( dy / d );
	int resz = ceil( dz / d );
	
	julia = _julia;
	
	julia_c.x = _julia_x;
	julia_c.y = _julia_y;
	julia_c.z = _julia_z;
	
	radiolaria = _radiolaria;
	radiolariaFactor = _radiolariaFactor;
	
	bailout = _bailout;

	phase.x = _phase_x;
	phase.y = _phase_y;
	
//	NSLog(@"start Mandelbulb output:%@, s_output:%@, resolutions:%d, iterations: %d, power:%f, bailout:%f, inner:%f, outer:%f,"
//				"julia:%d, jx:%f, jy:%f, jz:%f, rl:%d, rlf:%f",
//		  _output, _s_output, _resolutions, _iterations, _power, _bailout, _inner, _outer,
//		  _julia, julia_c.x, julia_c.y, julia_c.z,
//		  _radiolaria, _radiolariaFactor);
	
	// check
	__block int dot = 0;
	__block int iter = 0;
	
	char *grid;
	grid = (char*)malloc(sizeof(char)*resolutions*resolutions*resolutions);
	
	
	if (_useGCD) {
		
		int i,j;
		
		dispatch_queue_t queue = dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0);
		dispatch_group_t group = dispatch_group_create();
				
		for (i = 0;i < resx;i++) {
			for (j = 0;j < resy;j++) {
				
				dispatch_group_async(group, queue, ^{
					
					int k;
					Vec3D h;
					float h_len;
					double dist;
					
					for (k = 0;k < resz;k++) {
						h.x = _minx+(i*d);
						h.y = _miny+(j*d);
						h.z = _minz+(k*d);
						
						dist = DE( h, 4.0 );
						
						grid[ k + j*resz + i*resz*resy ] = 0;
						
						h_len = Vec3D_length( h );
						if (h_len < _outer && h_len > _inner) {
							if ( dist < MIN_EPSION ) {
								grid[ k + j*resz + i*resz*resy ] = 1;
								dot++;
							}
						}
						iter++;
					}
					
				});
			}
		}
		
		dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
		dispatch_release(group);
		
	} else {
		
		int i,j,k;
		
		Vec3D h;
		float h_len;
		double dist;
		
		
		for (i = 0;i < resx;i++) {
			for (j = 0;j < resy;j++) {
				for (k = 0;k < resz;k++) {
					h.x = _minx+(i*d);
					h.y = _miny+(j*d);
					h.z = _minz+(k*d);
					
					dist = DE( h, 4.0 );
					
					grid[iter] = 0;
					
					h_len = Vec3D_length( h );
					if (h_len < _outer && h_len > _inner) {
						if ( dist < MIN_EPSION ) {
							grid[iter] = 1;
							dot++;
						}
					}
					iter++;
				}
			}
		}
	}
	
	
	NSFileManager * fileManager = [NSFileManager defaultManager];
	
	if (_output != nil) {
		NSData * data;
		unsigned int *res_grid;
		int i;
		
		res_grid = (unsigned int*)malloc(sizeof(unsigned int)*resx*resy*resz);
		
		// Cheetah3D のスクリプトでバイナリを読む場合、ビッグエンディアンしか読めないようなので（おそらくバグ）
		// 強制的にビッグエンディアンに変換してテンポラリーファイルに書き込む
		for (i = 0;i < iter;i++) {
			res_grid[i] = NSSwapHostIntToBig( (unsigned int)grid[i] );
		}
		data = [NSData dataWithBytes:res_grid length:iter*sizeof(int)];
		
		[fileManager createFileAtPath:_output contents:data attributes:NULL];
		
		free( res_grid );
	}
	
	int num_triangles = 0;
	int s_iter = 0;
	
	/*
	if (_s_output == nil) {
		//_s_output = [NSString stringWithString:@"/tmp/frac.data"];
	}
	*/
	
	if (_s_output != nil) {
		MarchingCube * mc = [[MarchingCube alloc] initWithBytes:grid gridSizeX:resx sizeY:resy sizeZ:resz blurIter:_smooth useGCD:_useGCD];
		
		[mc setMin_x:_minx * _scale];
		[mc setMin_y:_miny * _scale];
		[mc setMin_z:_minz * _scale];
		[mc setGridSize: d * _scale];
		
		[mc recalculateWithIsovalue:_iso];
		
		num_triangles = [mc num_triangles];
		Triangle *triangles = [mc triangles];
		
		unsigned int *num_count;
		num_count = (unsigned int*)malloc(sizeof(unsigned int));
		NSSwappedFloat *points;
		points = (NSSwappedFloat*)malloc(sizeof(NSSwappedFloat)*num_triangles*3*3);
		
		// これも強制的にビッグエンディアンに変換
		num_count[0] = NSSwapHostIntToBig(num_triangles);
		//
		s_iter = 0;
		for (int t = 0; t < num_triangles; t++) {
			for (int i = 0; i < 3;i++) {
				points[s_iter] = NSSwapHostFloatToBig( triangles[t].points[i][0] );
				s_iter++;
				points[s_iter] = NSSwapHostFloatToBig( triangles[t].points[i][1] );
				s_iter++;
				points[s_iter] = NSSwapHostFloatToBig( triangles[t].points[i][2] );
				s_iter++;
			}
		}
		
		NSSwappedFloat tri_min[3];
		NSSwappedFloat tri_max[3];
		
		tri_min[0] = NSSwapHostFloatToBig( [mc tri_min_x] );
		tri_min[1] = NSSwapHostFloatToBig( [mc tri_min_y] );
		tri_min[2] = NSSwapHostFloatToBig( [mc tri_min_z] );
		
		tri_max[0] = NSSwapHostFloatToBig( [mc tri_max_x] );
		tri_max[1] = NSSwapHostFloatToBig( [mc tri_max_y] );
		tri_max[2] = NSSwapHostFloatToBig( [mc tri_max_z] );
		
//		NSLog(@"min {x:%.3f, y:%.3f, z:%.3f } max {x:%.3f, y:%.3f, z:%.3f}",
//							[mc tri_min_x], [mc tri_min_y], [mc tri_min_z], [mc tri_max_x], [mc tri_max_y], [mc tri_max_z]);
		
		NSMutableData * s_data = [[NSMutableData alloc] initWithLength:0];
		[s_data appendBytes:num_count length:sizeof( unsigned int )];
		[s_data appendBytes:tri_min length:(sizeof( NSSwappedFloat )*3)];
		[s_data appendBytes:tri_max length:(sizeof( NSSwappedFloat )*3)];
		[s_data appendBytes:points length:s_iter*sizeof(NSSwappedFloat)];
	
		[fileManager createFileAtPath:_s_output contents:s_data attributes:NULL];
		
		free(num_count);
		free(points);
		[mc release];
	}

	free(grid);
	
	NSLog( @"result Mandelbulb volume_size:%d, detected_set:%d; triangles:%d, vertices:%d", iter, dot, num_triangles, s_iter );
	
    return EXIT_SUCCESS;
}


@end
